<?php

namespace Viveum\Viveum\Model\Api;

use Viveum\Viveum\Api\WebhookReceiverInterface;
use Magento\Sales\Model\Order;
use Viveum\Viveum\Model\Method\AbstractMethod;

class WebhookReceiver implements WebhookReceiverInterface
{
	private $supported_payment_type = ['DB', 'PA'];
    private $request;
    private $responseController;
    private $abstractMethod;
    private $logger;
    private $generalFunction;

    private $payload;
    private $orderId;
    private $paymentMethod;
    private $order;

    public function __construct(
        \Magento\Framework\App\RequestInterface $request,
        \Viveum\Viveum\Controller\Payment\Response $responseController,
        \Viveum\Viveum\Helper\GeneralFunction $generalFunction
    )
    {
        $this->request = $request;
        $this->responseController = $responseController;
        $this->responseController->setNotRedirected(true);
        $this->abstractMethod = $this->responseController->createPaymentMethodObjectByPaymentMethod("viveum_creditcard");
        $this->logger = new \Viveum\Viveum\Helper\Logger("WebhookReceiver");
        $this->generalFunction = $generalFunction;
    }

    /**
     * {@inheritdoc}
     */
    public function receivePayload($encryptedBody)
    {
        $iv = $this->request->getHeader("X-Initialization-Vector");
        $authTag = $this->request->getHeader("X-Authentication-Tag");
        $keyFromConfiguration = $this->abstractMethod->getWebhookConfig("secret_for_encryption");

        $this->logger->info("receivePayload encryptedBody", $encryptedBody);
        $this->logger->info("receivePayload iv", $iv);
        $this->logger->info("receivePayload authTag", $authTag);
        $this->logger->info("receivePayload keyFromConfiguration", $keyFromConfiguration);

        $payload = $this->decryptPayload($encryptedBody, $keyFromConfiguration, $iv, $authTag);

		// sleep 10 second so it doesn't overlap with actual redirection
        sleep(10);

        if ( !$payload ) {
            $this->logger->error("receivePayload Unable to decrypt");

            return "Unable to decrypt";
        }

        return $this->handlePayloadFromPayon($payload);
    }

    private function decryptPayload($encryptedBody, $keyFromConfiguration, $iv, $authTag)
    {
		try {
			if ( $this->generalFunction->isPHPLowerThan71() && $this->generalFunction->isSodiumCryptoAeadAccessible() ) {
                $this->logger->info("decryptPayload isPHPLowerThan71 && isSodiumCryptoAeadAccessible");

                $key = hex2bin($keyFromConfiguration);
				$iv = hex2bin($iv);
				$cipher_text = hex2bin($encryptedBody . $authTag);

				$result = \Sodium\crypto_aead_aes256gcm_decrypt($cipher_text, NULL, $iv, $key);

			} else {
				$key = hex2bin($keyFromConfiguration);
				$iv = hex2bin($iv);
				$authTag = hex2bin($authTag);
				$cipher_text = hex2bin($encryptedBody);

				$result = openssl_decrypt($cipher_text, "aes-256-gcm", $key, OPENSSL_RAW_DATA, $iv, $authTag);
			}

			if ( gettype( $result ) == 'string' ) {
				$result = json_decode( $result, true );
			}

            $this->logger->info("decryptPayload result", $result);

            $this->payload = $result;

			return $result;

		} catch (Exception $e) {
            $this->logger->error("decryptPayload", $e->getMessage());
			return false;
		}
	}

	private function handlePayloadFromPayon($payload = array()) {
		if ( $payload['type'] !== 'PAYMENT' ) {
            $this->logger->info("handlePayloadFromPayon", "payload.type is not supported");
			return "payload.type is not supported";
		}

		$paymentResult = $payload['payload'];

		if ( ! in_array( $paymentResult['paymentType'], $this->supported_payment_type ) ) {
            $this->logger->info("handlePayloadFromPayon", "payload.paymentType is not supported");
			return "payload.paymentType is not supported";
		}

        // handle subscription
        if(isset($paymentResult['customParameters']['isSubscription']) ||
        (isset($paymentResult['source']) && $paymentResult['source'] == 'SCHEDULER'))
        {
            $this->responseController->processSubscriptionFromWebhook($paymentResult);
            return 'Order is processed';
        }


        $this->orderId = $paymentResult['customParameters']['orderId'];

        $this->order = $this->responseController->getOrderByIncerementId($this->orderId);

        if ( !in_array($this->order->getStatus(), [
            Order::STATE_PENDING_PAYMENT,
            Order::STATE_PAYMENT_REVIEW,
        ]) ) {
            $msg = "Order status is NOT PENDING";
            $this->logger->info("handlePayloadFromPayon", $msg);
            return $msg;
        }

		$this->paymentMethod = $this->responseController->createPaymentMethodObjectByPaymentMethod($paymentResult['customParameters']['paymentMethod']);

        $this->responseController->setPaymentMethodByCode($paymentResult['customParameters']['paymentMethod']);

        $this->logger->info("handlePayloadFromPayon orderId", $this->orderId);

        return $this->responseController->processPaymentResponse($paymentResult);
	}
}
