<?php
namespace Viveum\Viveum\Setup;

use Magento\Framework\Setup\InstallSchemaInterface;
use Magento\Framework\Setup\ModuleContextInterface;
use Magento\Framework\Setup\SchemaSetupInterface;
use Magento\Framework\DB\Ddl\Table;
use Magento\Framework\DB\Adapter\AdapterInterface;

/**
 * @codeCoverageIgnore
 */
class InstallSchema implements InstallSchemaInterface
{
    /**
     * install the Viveum schema
     * @param  SchemaSetupInterface   $setup
     * @param  ModuleContextInterface $context
     * @return void
     */
    public function install(SchemaSetupInterface $setup, ModuleContextInterface $context)
    {
        if (!$context->getVersion()) {
	        /**
	         * Prepare database for install
	         */
	        $setup->startSetup();

	        /**
	         * Create table 'viveum_payment_information'
	         */
	        $table = $setup->getConnection()->newTable(
	            $setup->getTable('viveum_payment_information')
	        )->addColumn(
	            'information_id',
	            \Magento\Framework\DB\Ddl\Table::TYPE_INTEGER,
	            null,
	            ['identity' => true, 'unsigned' => true, 'nullable' => false, 'primary' => true],
	            'Information Id'
	        )->addColumn(
	            'customer_id',
	            \Magento\Framework\DB\Ddl\Table::TYPE_INTEGER,
	            null,
	            ['unsigned' => true, 'nullable' => false],
	            'Customer Id'
	        )->addColumn(
	            'payment_group',
	            \Magento\Framework\DB\Ddl\Table::TYPE_TEXT,
	            6,
	            ['nullable' => false],
	            'Payment Group'
	        )->addColumn(
	            'server_mode',
	            \Magento\Framework\DB\Ddl\Table::TYPE_TEXT,
	            4,
	            ['nullable' => false],
	            'Server Mode'
	        )->addColumn(
	            'channel_id',
	            \Magento\Framework\DB\Ddl\Table::TYPE_TEXT,
	            32,
	            ['nullable' => false],
	            'Channel Id'
	        )->addColumn(
	            'registration_id',
	            \Magento\Framework\DB\Ddl\Table::TYPE_TEXT,
	            32,
	            ['nullable' => false],
	            'Registration Id'
	        )->addColumn(
	            'brand',
	            \Magento\Framework\DB\Ddl\Table::TYPE_TEXT,
	            100,
	            ['nullable' => false],
	            'Brand'
	        )->addColumn(
	            'holder',
	            \Magento\Framework\DB\Ddl\Table::TYPE_TEXT,
	            100,
	            ['nullable' => false],
	            'Holder'
	        )->addColumn(
	            'email',
	            \Magento\Framework\DB\Ddl\Table::TYPE_TEXT,
	            60,
	            ['nullable' => false],
	            'Email'
	        )->addColumn(
	            'last_4digits',
	            \Magento\Framework\DB\Ddl\Table::TYPE_TEXT,
	            4,
	            ['nullable' => false],
	            'Last 4 Digits'
	        )->addColumn(
	            'expiry_month',
	            \Magento\Framework\DB\Ddl\Table::TYPE_TEXT,
	            2,
	            ['nullable' => false],
	            'Expiry Month'
	        )->addColumn(
	            'expiry_year',
	            \Magento\Framework\DB\Ddl\Table::TYPE_TEXT,
	            4,
	            ['nullable' => false],
	            'Expiry Year'
	        )->addColumn(
	            'payment_default',
	            \Magento\Framework\DB\Ddl\Table::TYPE_BOOLEAN,
	            null,
	            ['nullable' => false, 'default' => '0'],
	            'Payment Default'
	        );
	        $setup->getConnection()->createTable($table);

            $this->createPlanTable($setup);
            $this->createSubscriptionTable($setup);
            $this->createSubscriptionPaymentTable($setup);

	        /**
	         * Prepare database after install
	         */
	        $setup->endSetup();
	    }
    }

    public function createPlanTable(SchemaSetupInterface $setup)
    {
        $table = $setup->getConnection()->newTable($setup->getTable('viveum_plans'));

        $table->addColumn(
            'plan_id',
            Table::TYPE_INTEGER,
            null,
            [
                'identity' => true,
                'unsigned' => true,
                'primary' => true,
                'nullable' => false,
            ]
        )
        ->addColumn(
            'magento_product_id',
            Table::TYPE_INTEGER,
            null,
            [
                'nullable' => false,
                'comment' => 'Magento Product ID'
            ]
        )
        ->addColumn(
            'plan_name',
            Table::TYPE_TEXT,
            255,
            [
                'nullable' => false,
                'comment' => 'Plan Name',
            ]
        )
        ->addColumn(
            'plan_description',
            Table::TYPE_TEXT,
            255,
            [
                'nullable' => false,
                'comment' => 'Plan Description'
            ]
        )
        ->addColumn(
            'plan_type',
            Table::TYPE_TEXT,
            50,
            [
                'nullable' => false,
                'comment' => 'daily, weekly, monthly, yearly',
            ]
        )
        ->addColumn(
            'plan_number_of_payment',
            Table::TYPE_INTEGER,
            null,
            [
                'nullable' => false,
                'comment' => 'Total number of payments'
            ]
        )
        ->addColumn(
            'plan_initial_amount',
            Table::TYPE_DECIMAL,
            null,
            [
                'nullable' => false,
                'default' => 0,
                'comment' => 'Initial amount payment',
            ]
        )
        ->addColumn(
            'plan_charge_amount',
            Table::TYPE_DECIMAL,
            null,
            [
                'nullable' => false,
                'default' => 0,
                'comment' => 'Subscription charge'
            ]
        )
        ->addColumn(
            'plan_status',
            Table::TYPE_INTEGER,
            null,
            [
                'nullable' => false,
                'default' => 1,
                'comment' => '1: Enable, 2: Disabled',
            ]
        )
        ->addColumn(
            'created_at',
            Table::TYPE_TIMESTAMP,
            null,
            [
                'nullable' => false,
                'default' => Table::TIMESTAMP_INIT,
            ]
        )
        ->addColumn(
            'updated_at',
            Table::TYPE_TIMESTAMP,
            null,
            [
                'nullable' => false,
                'default' => Table::TIMESTAMP_INIT_UPDATE,
            ]
        )
        ->addIndex(
            'plan',
            ['plan_id'],
            [
                'type' => AdapterInterface::INDEX_TYPE_UNIQUE,
                'nullable' => false,
            ]
        )
        ->addIndex(
            'plan_product',
            ['plan_id', 'plan_type', 'magento_product_id', 'plan_name'],
            ['type' => AdapterInterface::INDEX_TYPE_INDEX]
        );

        $setup->getConnection()->createTable($table);
    }


    private function createSubscriptionTable($setup)
    {
        $table = $setup->getConnection()->newTable($setup->getTable('viveum_subscriptions'));

        $table
            ->addColumn(
                'entity_id',
                Table::TYPE_INTEGER,
                null,
                [
                    'identity' => true,
                    'unsigned' => true,
                    'primary' => true,
                    'nullable' => false,
                ]
            )
            ->addColumn(
                'registration_id',
                Table::TYPE_TEXT,
                200,
                [
                    'nullable' => false,
                    'comment' => 'RG Unique ID',
                ]
            )
            ->addColumn(
                'subscription_id',
                Table::TYPE_TEXT,
                200,
                [
                    'unique'   => true,
                    'nullable' => false,
                    'comment' => 'SD Unique ID'
                ]
            )
            ->addColumn(
                'plan_id',
                Table::TYPE_TEXT,
                100,
                [
                    'nullable' => false,
                    'comment' => 'Plan ID'
                ]
            )
            ->addColumn(
                'quote_id',
                Table::TYPE_INTEGER,
                [
                    'nullable' => true,
                    'comment' => 'Magento quote id'
                ]
            )
            ->addColumn(
                'product_id',
                Table::TYPE_INTEGER,
                [
                    'nullable' => true,
                    'comment' => 'Magento product id'
                ]
            )
            ->addColumn(
                'product_qty',
                Table::TYPE_INTEGER,
                [
                    'nullable' => false,
                    'comment' => 'Product Qty'
                ]
            )
            ->addColumn(
                'magento_user_id',
                Table::TYPE_INTEGER,
                null,
                [
                    'nullable' => true,
                    'comment' => 'Magento user id'
                ]
            )
            ->addColumn(
                'customer_firstname',
                Table::TYPE_TEXT,
                200,
                [
                    'nullable' => true,
                    'comment' => 'Customer First Name',
                ]
            )
            ->addColumn(
                'customer_lastname',
                Table::TYPE_TEXT,
                200,
                [
                    'nullable' => true,
                    'comment' => 'Customer Last Name',
                ]
            )
            ->addColumn(
                'customer_email',
                Table::TYPE_TEXT,
                200,
                [
                    'nullable' => true,
                    'comment' => 'Customer Email',
                ]
            )
            ->addColumn(
                'transaction_id',
                Table::TYPE_TEXT,
                200,
                [
                    'nullable' => false,
                    'comment' => 'Merchant Transaction ID',
                    'unique' => true,
                ]
            )
            ->addColumn(
                'status',
                Table::TYPE_TEXT,
                30,
                [
                    'nullable' => false,
                    'comment' => 'Subscription status'
                ]
            )
            ->addColumn(
                'cancel_by',
                Table::TYPE_TEXT,
                30,
                [
                    'nullable' => false,
                    'comment' => 'Cancel by user or admin'
                ]
            )
            ->addColumn(
                'paid_count',
                Table::TYPE_INTEGER,
                30,
                [
                    'nullable' => false,
                    'default' => 0,
                    'comment' => 'Total subscription paid count'
                ]
            )
            ->addColumn(
                'subscription_charge',
                Table::TYPE_TEXT,
                30,
                [
                    'nullable' => false,
                    'default' => 0,
                    'comment' => 'Total charge per subscription.'
                ]
            )
            ->addColumn(
                'payment_method',
                Table::TYPE_TEXT,
                100,
                [
                    'nullable' => false,
                    'default' => 0,
                    'comment' => 'Payment method'
                ]
            )
            ->addColumn(
                'start_at',
                \Magento\Framework\DB\Ddl\Table::TYPE_TIMESTAMP,
                null,
                [
                    'nullable' => true,
                    'comment' => 'Subscription start date'
                ]
            )
            ->addColumn(
                'next_charge_at',
                \Magento\Framework\DB\Ddl\Table::TYPE_TIMESTAMP,
                null,
                [
                    'nullable' => true,
                    'comment' => 'Next charge date'
                ]
            )
            ->addColumn(
                'created_at',
                \Magento\Framework\DB\Ddl\Table::TYPE_TIMESTAMP,
                null,
                [
                    'nullable' => false,
                    'default' => Table::TIMESTAMP_INIT,
                    'comment' => 'Created at'
                ]
            )
            ->addColumn(
                'updated_at',
                Table::TYPE_TIMESTAMP,
                null,
                [
                    'nullable' => false,
                    'default' => Table::TIMESTAMP_INIT_UPDATE,
                ]
            )
            ->addIndex(
                'registration_id',
                ['registration_id'],
                [
                    'type'      => AdapterInterface::INDEX_TYPE_INDEX,
                    'nullable'  => false,
                ]
            )
            ->addIndex(
                'transaction_id',
                ['transaction_id'],
                [
                    'type'      => AdapterInterface::INDEX_TYPE_UNIQUE,
                    'nullable'  => false,
                    'unique' => true,
                ]
            )
            ->addIndex(
                'subscription_id',
                ['subscription_id'],
                [
                    'type'      => AdapterInterface::INDEX_TYPE_UNIQUE,
                    'nullable'  => false,
                    'unique'    => true,
                ]
            )
            ->addIndex(
                'plan_id',
                ['plan_id'],
                [
                    'type'      => AdapterInterface::INDEX_TYPE_INDEX,
                ]
            )
            ->addIndex(
                'subscription_status',
                ['subscription_id','status'],
                [
                    'type'      => AdapterInterface::INDEX_TYPE_INDEX,
                ]
            )
            ->addIndex(
                'product_id',
                ['subscription_id','product_id'],
                [
                    'type'      => AdapterInterface::INDEX_TYPE_INDEX,
                ]
            )
            ->addIndex(
                'subscription_user',
                ['subscription_id','magento_user_id'],
                [
                    'type'      => AdapterInterface::INDEX_TYPE_INDEX,
                ]
            )
        ;

        $setup->getConnection()->createTable($table);

    }


    private function createSubscriptionPaymentTable($setup)
    {
        $table = $setup->getConnection()->newTable($setup->getTable('viveum_subscription_payments'));

        $table
            ->addColumn(
                'entity_id',
                Table::TYPE_INTEGER,
                null,
                [
                    'identity' => true,
                    'unsigned' => true,
                    'primary' => true,
                    'nullable' => false,
                ]
            )
            ->addColumn(
                'subscription_id',
                Table::TYPE_TEXT,
                200,
                [
                    'nullable' => false,
                    'comment' => 'SD Unique ID'
                ]
            )
            ->addColumn(
                'payment_id',
                Table::TYPE_TEXT,
                200,
                [
                    'unique' => true,
                    'nullable' => false,
                    'comment' => 'DB Unique ID'
                ]
            )
            ->addColumn(
                'transaction_id',
                Table::TYPE_TEXT,
                200,
                [
                    'nullable' => false,
                    'comment' => 'Merchant Transaction ID'
                ]
            )
            ->addColumn(
                'from_webhook',
                Table::TYPE_INTEGER,
                30,
                [
                    'nullable' => false,
                    'default' => 0,
                    'comment' => '1 if from webhook'
                ]
            )
            ->addColumn(
                'amount',
                Table::TYPE_TEXT,
                30,
                [
                    'nullable' => false,
                    'default' => 0,
                    'comment' => 'Total charge per subscription.'
                ]
            )
            ->addColumn(
                'status',
                Table::TYPE_TEXT,
                100,
                [
                    'nullable' => false,
                    'comment' => 'Status payment, success or failed'
                ]
            )
            ->addColumn(
                'created_at',
                \Magento\Framework\DB\Ddl\Table::TYPE_TIMESTAMP,
                null,
                [
                    'nullable' => false,
                    'default' => Table::TIMESTAMP_INIT,
                    'comment' => 'Created at'
                ]
            )
            ->addColumn(
                'updated_at',
                Table::TYPE_TIMESTAMP,
                null,
                [
                    'nullable' => false,
                    'default' => Table::TIMESTAMP_INIT_UPDATE,
                ]
            )
            ->addIndex(
                'transaction_id',
                ['transaction_id'],
                [
                    'type'      => AdapterInterface::INDEX_TYPE_INDEX,
                    'nullable'  => false,
                ]
            )
            ->addIndex(
                'subscription_id',
                ['subscription_id'],
                [
                    'type'      => AdapterInterface::INDEX_TYPE_INDEX,
                    'nullable'  => false,
                ]
            )
            ->addIndex(
                'payment_id',
                ['payment_id'],
                [
                    'type'      => AdapterInterface::INDEX_TYPE_UNIQUE,
                    'nullable'  => false,
                ]
            )
        ;

        $setup->getConnection()->createTable($table);

    }
}
