<?php

namespace Viveum\Viveum\Controller;

class Payment extends \Magento\Framework\App\Action\Action
{
    protected $quote = false;
    public $order = false;
    protected $checkoutSession;
    protected $localeResolver;
    protected $resultPageFactory;
    public $helperPayment;
    protected $information;
    public $paymentMethod;
    protected $quoteManagement;
    protected $customer;
    protected $klarnaCartItemFlags = '32';
    public $viveum = 'viveum';
    public $curl;
    protected $logger;
    protected $_modelCart;
    protected $subscription;
    protected $subscriptionPayment;

    /**
     *
     * @param \Magento\Framework\App\Action\Context                    $context
     * @param \Magento\Checkout\Model\Session                          $checkoutSession
     * @param \Magento\Framework\Locale\ResolverInterface              $localeResolver
     * @param \Magento\Framework\View\Result\PageFactory               $resultPageFactory
     * @param \Viveum\Viveum\Helper\Payment               $helperPayment
     * @param \Viveum\Viveum\Model\Payment\Information $information
     * @param \Viveum\Viveum\Model\Customer\Customer   $customer
     * @param \Viveum\Viveum\Model\Subscription\Subscription   $subscription
     * @param \Viveum\Viveum\Model\Subscription\SubscriptionPayment   $subscriptionPayment
     */
    public function __construct(
        \Magento\Framework\App\Action\Context $context,
        \Magento\Checkout\Model\Session $checkoutSession,
        \Magento\Framework\Locale\ResolverInterface $localeResolver,
        \Magento\Framework\View\Result\PageFactory $resultPageFactory,
        \Viveum\Viveum\Helper\Payment $helperPayment,
        \Viveum\Viveum\Helper\Curl $curl,
        \Viveum\Viveum\Model\Payment\Information $information,
        \Viveum\Viveum\Model\Customer\Customer $customer,
        \Viveum\Viveum\Model\Subscription\Subscription $subscription,
        \Viveum\Viveum\Model\Subscription\SubscriptionPayment $subscriptionPayment
    ) {
        parent::__construct($context);
        $this->checkoutSession = $checkoutSession;
        $this->localeResolver = $localeResolver;
        $this->resultPageFactory = $resultPageFactory;
        $this->helperPayment = $helperPayment;
        $this->curl = $curl;
        $this->information = $information;
        $this->customer = $customer;
        $this->subscription = $subscription;
        $this->subscriptionPayment = $subscriptionPayment;
        $this->logger = new \Viveum\Viveum\Helper\Logger("Controller\Payment");
    }

    /**
     *
     * @return void
     */
    public function execute()
    {
        $this->setPaymentMethod();

        if ($this->isSubscriptionOrder()) {
            if ($this->paymentMethod->getCode() == 'viveum_paypal') {
                $this->paymentMethod = $this->createPaymentMethodObjectByPaymentMethod('paypalsaved');
            }

            if ($this->paymentMethod->getCode() == 'viveum_creditcard') {
                $this->paymentMethod = $this->createPaymentMethodObjectByPaymentMethod('ccsaved');
            }

            if (
                $this->paymentMethod->getCode() == 'viveum_paypalsaved' &&
                $this->order->getPayment()->getViveum_user_consent() != 1
            ) {
                $this->redirectError('Please save your PayPal data');
            }
        }

        if ($this->paymentMethod->isServerToServer()) {
            $isAvailableMethodByAddress = $this->isAvailableMethodByAddress();
            if ($isAvailableMethodByAddress) {
                return $this->processServerToServer();
            } else {
                $this->redirectError('ERROR_MESSAGE_BILLING_SHIPPING_NOTSAME');
            }
        } else {
            return $this->processCheckout();
        }
    }

    /**
     * set paymentMethod
     * @return void
     */
    protected function setPaymentMethod()
    {
        $this->order = $this->getOrder();
        $this->paymentMethod = $this->order->getPayment()->getMethodInstance();
    }

    /**
     * check if a customer has a billing address same as a shipping address then the klarna payment method is available
     * @return boolean
     */
    protected function isAvailableMethodByAddress()
    {
        if ($this->isKlarnaPayment()) {
            $isShippingAddressSameAsBilling = $this->paymentMethod->isShippingAddressSameAsBilling();

            if (!$isShippingAddressSameAsBilling) {
                return false;
            }
        }
        return true;
    }

    /**
     * process checkout for server to server method
     * @return object
     */
    protected function processServerToServer()
    {

        if ($this->isSubscriptionOrder()) {

            $paymentParameters = $this->getSubscriptionRegisterParameters();
            $this->logger->info("processServerToServer paymentParameters", $paymentParameters);

            $checkoutParameters = array_merge(
                $this->paymentMethod->getCredentials(),
                $this->getTransactionParameters(),
                $paymentParameters,
                $this->getServerToServerParameters()
            );
        } else {

            $checkoutParameters = array_merge_recursive(
                $this->getPaymentParameters(),
                $this->getServerToServerParameters()
            );
        }

        $paymentMethod = $this->paymentMethod;
        $initialPaymentResponse = $this->helperPayment->initializeServerToServerPayment($checkoutParameters);
        if ($initialPaymentResponse['isValid']) {
            if (isset($initialPaymentResponse['response']['redirect']['url'])) {
                $resultPageFactory = $this->resultPageFactory->create();
                // Add page title
                $resultPageFactory->getConfig()->getTitle()->set($paymentMethod->getTitle());
                $blockViveum = $resultPageFactory->getLayout()->getBlock('viveumPaymentForm');
                $blockViveum->setTemplate('Viveum_Viveum::payment/responseredirect.phtml');
                $blockViveum->setInitialPaymentResponse($initialPaymentResponse['response']);

                return $resultPageFactory;
            }
        }
        $this->redirectError('ERROR_GENERAL_REDIRECT');
    }

    /**
     * process checkout for copy and pay method
     * @return object $resultPageFactory;
     */
    protected function processCheckout()
    {
        if ($this->order->getState() == "processing") {
            $this->_redirect('checkout/onepage/success');
        }

        if ($this->isSubscriptionOrder()) {
            $paymentParameters = $this->getSubscriptionRegisterParameters();
            // $paymentParameters['registrations'] = false;
            $this->logger->info("paymentParameters subscription: ", $paymentParameters);
            $checkoutResult = $this->helperPayment->getCheckoutResult($paymentParameters);
        } else {
            $paymentParameters = $this->getPaymentParameters();
            $checkoutResult = $this->helperPayment->getCheckoutResult($paymentParameters);
        }


        if (!$checkoutResult['isValid']) {
            $this->redirectError($checkoutResult['response']);
        } elseif (!isset($checkoutResult['response']['id'])) {
            $this->redirectError('ERROR_GENERAL_REDIRECT');
        } else {
            $paymentWidgetUrl  = $this->helperPayment->getPaymentWidgetUrl(
                $paymentParameters['serverMode'],
                $checkoutResult['response']['id']
            );

            $paymentWidgetContent = $this->curl->getPaymentWidgetContent(
                $paymentWidgetUrl,
                $paymentParameters['proxy'],
                $paymentParameters['serverMode'],
                $this->helperPayment->setHeaders($paymentParameters)
            );

            if (
                !$paymentWidgetContent['isValid'] ||
                strpos($paymentWidgetContent['response'], 'errorDetail') !== false
            ) {
                $this->redirectError('ERROR_GENERAL_REDIRECT');
            }

            $resultPageFactory = $this->resultPageFactory->create();
            // Add page title
            $resultPageFactory->getConfig()->getTitle()->set($this->paymentMethod->getTitle());
            $this->addBreadCrumbs($resultPageFactory);

            $cancelUrl = $this->_url->getUrl('checkout/cart', ['_secure' => true]);
            $responseUrl = $this->_url->getUrl(
                'viveum/payment/response',
                [
                    'payment_method' => $this->paymentMethod->getCode(),
                    '_secure' => true
                ]
            );
            $this->setPageAsset($resultPageFactory);

            $blockViveum = $resultPageFactory->getLayout()->getBlock('viveumPaymentForm');
            if ($this->paymentMethod->getCode() == 'viveum_paypalsaved') {
                $paymentParameters['registrations'] = $this->getRegisteredPaypal();
                $this->setPaypalSavedBlock($blockViveum);
            }
            $blockViveum->setPaymentCode($this->paymentMethod->getCode());
            $blockViveum->setBrand($this->paymentMethod->getBrand());
            $blockViveum->setIsRecurringPayment($this->paymentMethod->isRecurringPayment());
            $blockViveum->setRegistrations($paymentParameters['registrations']);
            $blockViveum->setWidgetStyle($this->paymentMethod->getWidgetStyle());
            $blockViveum->setLang($this->paymentMethod->getLangCode());
            $blockViveum->setTestMode($this->paymentMethod->getTestMode());
            $blockViveum->setCancelUrl($cancelUrl);
            $blockViveum->setResponseUrl($responseUrl);
            $blockViveum->setPaymentWidgetUrl($paymentWidgetUrl);
            $blockViveum->setMerchantLocation($this->paymentMethod->getGeneralConfig('merchant_location'));
            $blockViveum->setChannelId($this->paymentMethod->getConfigData('channel_id'));
            $blockViveum->setGoogleMerchantId($this->paymentMethod->getConfigData('google_merchant_id'));
            $blockViveum->setAppleMerchantId($this->paymentMethod->getConfigData('apple_merchant_id'));
            $blockViveum->setAuthMethod($this->paymentMethod->getConfigData('auth_method'));
            $blockViveum->setButtonStyle($this->paymentMethod->getConfigData('button_style'));
            $blockViveum->setCardType($this->paymentMethod->getConfigData('card_selection'));
            $blockViveum->setShopName($this->paymentMethod->getConfigData('shop_name'));
            $blockViveum->setIsSubscription($this->isSubscriptionOrder() ? true : false);
            $blockViveum->setIsSavePayment($this->order->getPayment()->getViveum_user_consent() == 1 ? true : false);

            return $resultPageFactory;
        }
    }

    /**
     * set a paypal saved block
     * @param object $blockViveum
     */
    protected function setPaypalSavedBlock($blockViveum)
    {
        $blockViveum->setTemplate('Viveum_Viveum::payment/paypalsaved.phtml');
        $controller = 'viveum/payment/response';
        $repeatedPaypalresponseUrl = $this->_url->getUrl(
            $controller,
            [
                'payment_method' => $this->paymentMethod->getCode(),
                'repeated_paypal' => true,
                '_secure' => true
            ]
        );
        $formKey = $this->_objectManager->create('Magento\Framework\Data\Form\FormKey');
        $blockViveum->setFormKey($formKey->getFormKey());
        $blockViveum->setOrderId($this->getOrderIncrementId());
        $blockViveum->setRepeatedPaypalResponseUrl($repeatedPaypalresponseUrl);
    }

    /**
     * add bread crumbs
     * @param object $resultPageFactory
     */
    protected function addBreadCrumbs($resultPageFactory)
    {
        // Add breadcrumb
        /** @var \Magento\Theme\Block\Html\Breadcrumbs */
        $breadcrumbs = $resultPageFactory->getLayout()->getBlock('breadcrumbs');
        $breadcrumbs->addCrumb(
            'home',
            [
                'label' => __('Home'),
                'title' => __('Home'),
                'link' => $this->_url->getUrl('')
            ]
        );
        $breadcrumbs->addCrumb(
            $this->paymentMethod->getCode(),
            [
                'label' => $this->paymentMethod->getTitle(),
                'title' => $this->paymentMethod->getTitle()
            ]
        );
    }

    /**
     * set a page asset
     * @param object $resultPageFactory
     */
    protected function setPageAsset($resultPageFactory)
    {
        $widgetStyle = $this->paymentMethod->getWidgetStyle();
        if ($widgetStyle == 'custom') {
            switch ($this->paymentMethod->getCode()) {
                case 'viveum_ccsaved':
                case 'viveum_creditcard':
                case 'viveum_ddsaved':
                case 'viveum_directdebit':
                    $resultPageFactory->getConfig()->addPageAsset(
                        'Viveum_Viveum/css/payment_form_custom.css'
                    );
                    break;
                default:
                    $resultPageFactory->getConfig()->addPageAsset('Viveum_Viveum/css/payment_form.css');
                    break;
            }
        } else {
            $resultPageFactory->getConfig()->addPageAsset('Viveum_Viveum/css/payment_form.css');
        }
    }

    /**
     * set a page asset for my payment information page
     * @param [type] $resultPageFactory
     */
    protected function setPageAssetRecurring($resultPageFactory)
    {
        $widgetStyle = $this->paymentMethod->getWidgetStyle();
        if ($widgetStyle == 'custom') {
            $resultPageFactory->getConfig()->addPageAsset(
                'Viveum_Viveum/css/recurring_form_custom.css'
            );
        } else {
            $resultPageFactory->getConfig()->addPageAsset('Viveum_Viveum/css/recurring_form.css');
        }
    }

    /**
     * check if the Viveum payment methods are available
     * @param  string  $paymentMethod
     * @return boolean
     */
    public function isViveumMethod($paymentMethod)
    {
        $viveumMethod = strpos($paymentMethod, $this->viveum);
        if ($viveumMethod !== false) {
            return true;
        }
        return false;
    }

    /**
     * create a payment method object
     * @param  string $paymentMethod
     * @return object
     */
    public function createPaymentMethodObjectByPaymentMethod($paymentMethod)
    {
        $paymentMethodNameSpace = 'Viveum\Viveum\Model\Method\\' .
            $this->getPaymentMethodClassName($paymentMethod);
        return $this->_objectManager->create($paymentMethodNameSpace);
    }

    /**
     * get a payment method class name
     * @param  string $paymentMethod
     * @return string
     */
    public function getPaymentMethodClassName($paymentMethod)
    {
        $methods = [
            'ccsaved' => 'CCSaved',
            'creditcard' => 'CreditCard',
            'ddsaved' => 'DDSaved',
            'directdebit' => 'DirectDebit',
            'giropay' => 'Giropay',
            'paypal' => 'Paypal',
            'paypalsaved' => 'PaypalSaved',
            'klarnaobt' => 'Klarnaobt',
            'googlepay' => 'Googlepay',
            'applepay' => 'Applepay',
            'eps' => 'EPS',
        ];

        $code = str_replace('viveum_', '', $paymentMethod);

        if (isset($methods[$code])) {
            return $methods[$code];
        }

        return 'AbstractMethod';
    }

    /**
     * get payment parameters
     * @return array
     */
    protected function getPaymentParameters()
    {
        $paymentParameters = [];
        $paymentParameters = array_merge_recursive(
            $this->paymentMethod->getCredentials(),
            $this->getTransactionParameters(),
            $this->getCustomerInformationByOrder(),
            $this->getCustomerAddressByOrder(),
            $this->getCartItemsParameters(),
            $this->getCC3DParameters(),
            $this->getCustomParameters(),
            $this->getRegistrationParameters()
        );

        return $paymentParameters;
    }

    /**
     * get payment parameters
     * @return array
     */
    protected function getSubscriptionPaymentParameters()
    {
        $paymentParameters = [];
        $paymentParameters = array_merge_recursive(
            $this->paymentMethod->getCredentials(),
            $this->getTransactionParameters(),
            $this->getCustomerInformationByOrder(),
            $this->getCustomerAddressByOrder(),
            $this->getCartItemsParameters(),
            $this->getCC3DParameters(),
            $this->getCustomParameters(),
            $this->getRegistrationParameters()
        );

        return $paymentParameters;
    }

    /**
     * get the server to server parameters
     * @return array
     */
    protected function getServerToServerParameters()
    {
        $serverToServerParameters = [];

        $paymentMethod = $this->paymentMethod;

        $serverToServerParameters['paymentBrand'] = $paymentMethod->getBrand();
        $serverToServerParameters['shopperResultUrl'] =
            $this->_url->getUrl(
                'viveum/payment/response',
                [
                    'payment_method' => $paymentMethod->getCode(),
                    'server_to_server' => $paymentMethod::ASYNCHRONOUS,
                    '_secure' => true
                ]
            );

        return $serverToServerParameters;
    }

    /**
     * get registration parameters
     * @return array
     */
    protected function getRegistrationParameters()
    {
        $registrationParameters = [];

        $isRecurring = $this->paymentMethod->isRecurring();

        $registrationParameters['registrations'] = [];
        if ($isRecurring && $this->paymentMethod->isRecurringPayment()) {
            $informationParamaters = $this->getInformationParamaters();
            if ($this->paymentMethod->getCode() == 'viveum_paypalsaved' && $this->order->getPayment()->getViveum_user_consent() == 1) {
                $registrationParameters['createRegistration'] = 'true';
            } else {
                $paymentInformation = $this->information->getPaymentInformation($informationParamaters);
                if (!empty($paymentInformation)) {
                    foreach ($paymentInformation as $key => $registeredPayment) {
                        $registrationParameters['registrations'][$key] = $registeredPayment['registration_id'];
                    }
                }
            }
        }

        return $registrationParameters;
    }

    /**
     * get registered paypal accounts
     * @return array
     */
    protected function getRegisteredPaypal()
    {
        $isRecurring = $this->paymentMethod->isRecurring();

        $registeredPaypal = [];
        if (
            $isRecurring
            && $this->paymentMethod->isRecurringPayment()
            && $this->paymentMethod->getCode() == 'viveum_paypalsaved'
        ) {
            $informationParamaters = $this->getInformationParamaters();
            $paymentInformation = $this->information->getPaymentInformation($informationParamaters);
            if (!empty($paymentInformation)) {
                foreach ($paymentInformation as $key => $registeredPayment) {
                    $registeredPaypal[$key]['registrationId'] = $registeredPayment['registration_id'];
                    $registeredPaypal[$key]['email'] = $registeredPayment['email'];
                }
            }
        }

        return $registeredPaypal;
    }

    /**
     * get order increment id
     * @return string
     */
    protected function getOrderIncrementId()
    {
        return $this->order->getIncrementId();
    }

    /**
     * get order currency
     * @return string
     */
    protected function getOrderCurrency()
    {
        return $this->order->getOrderCurrencyCode();
    }

    /**
     * get transaction parameters
     * @return array
     */
    public function getTransactionParameters()
    {
        $transactionParameters = [];
        if ($this->isSendPaymentTypeParameter()) {
            $transactionParameters['paymentType'] = $this->paymentMethod->getPaymentType();
        }
        $transactionParameters['amount'] = $this->order->getGrandTotal();
        $transactionParameters['currency'] = $this->getOrderCurrency();
        $transactionParameters['transactionId'] = $this->getOrderIncrementId();

        $this->checkoutSession->setTransactionId($transactionParameters['transactionId']);
        $transactionParameters['customParameters']['orderId'] = $this->getOrderIncrementId();

        $this->checkoutSession->setOrderId($transactionParameters['customParameters']['orderId']);
        $transactionParameters['customParameters']['paymentMethod'] =  $this->paymentMethod->getCode();
        $this->checkoutSession->setLastSessionId($this->checkoutSession->getSessionId());

        if ($this->isSubscriptionOrder()) {
            $transactionParameters['customParameters']['isSubscription'] = 'true';
        }

        $this->logger->info('transaction parameter:', $transactionParameters);
        return $transactionParameters;
    }

    /**
     * get a customer information
     * @return array
     */
    protected function getCustomerInformationByOrder()
    {
        $customerInformation = [];
        $customerInformation['customer']['email'] = $this->order->getBillingAddress()->getEmail();
        $customerInformation['customer']['firstName'] = $this->order->getBillingAddress()->getFirstname();
        $customerInformation['customer']['lastName'] = $this->order->getBillingAddress()->getLastname();

        $quoteCartRepository = $this->_objectManager->create('Magento\Quote\Api\CartRepositoryInterface');

        $cartId = $this->getQuote()->getId();

        $quoteCart = $quoteCartRepository->getActive($cartId);
        if (!$quoteCart->getCheckoutMethod()) {
            $quoteCart->setCheckoutMethod('guest');
            $quoteCartRepository->save($quoteCart);
        }

        if ($this->isKlarnaPayment() || $this->isEasyCreditPayment()) {
            if (!$this->customer->getGender()) {
                $customerInformation['customer']['sex'] = $_COOKIE['gender'];
            } else {
                $customerInformation['customer']['sex'] = $this->customer->getGender();
            }
            if (!$this->isEasyCreditPayment()) {
                $customerInformation['customer']['birthdate'] = $this->customer->getDob();
            }
            $customerInformation['customer']['phone'] = $this->order->getBillingAddress()->getTelephone();
        }

        return $customerInformation;
    }

    /**
     * get a customer address
     * @return array
     */
    protected function getCustomerAddressByOrder()
    {
        $customerAddresss = [];
        $customerAddresss['billing']['street'] = implode(' ', $this->order->getBillingAddress()->getStreet());
        $customerAddresss['billing']['city'] = $this->order->getBillingAddress()->getCity();
        $customerAddresss['billing']['zip'] = $this->order->getBillingAddress()->getPostcode();
        $customerAddresss['billing']['state'] = $this->order->getBillingAddress()->getRegion();
        $customerAddresss['billing']['countryCode'] = $this->order->getBillingAddress()->getCountryId();

        if ($this->isKlarnaPayment() || $this->isEasyCreditPayment()) {
            $customerAddresss['shipping']['street'] = implode(' ', $this->order->getShippingAddress()->getStreet());
            $customerAddresss['shipping']['city'] = $this->order->getShippingAddress()->getCity();
            $customerAddresss['shipping']['zip'] = $this->order->getShippingAddress()->getPostcode();
            $customerAddresss['shipping']['state'] = $this->order->getShippingAddress()->getRegion();
            $customerAddresss['shipping']['countryCode'] = $this->order->getShippingAddress()->getCountryId();
        }

        return $customerAddresss;
    }

    /**
     * is Klarna payment
     *
     * @return boolean
     */
    public function isKlarnaPayment()
    {
        if (
            $this->paymentMethod->getCode() == 'viveum_klarnapaylater' ||
            $this->paymentMethod->getCode() == 'viveum_klarnasliceit'
        ) {
            return true;
        }
        return false;
    }

    /**
     * get klarna parameters
     * @return array
     */
    protected function getKlarnaParameters()
    {
        $klarnaParameters = [];

        if ($this->isKlarnaPayment()) {
            $klarnaParameters['customParameters']['klarnaCartItem1Flags'] = $this->klarnaCartItemFlags;

            if ($this->paymentMethod->getCode() == 'viveum_klarnasliceit') {
                $klarnaParameters['customParameters']['klarnaPclassFlag'] =
                    $this->paymentMethod->getConfigData('pclass_id');
            }
        }

        return $klarnaParameters;
    }

    /**
     * is Easycredit payment
     *
     * @return boolean
     */
    public function isEasyCreditPayment()
    {
        if ($this->paymentMethod->getCode() == 'viveum_easycredit') {
            return true;
        }
        return false;
    }

    /**
     * get easy credit parameters
     *
     * @return array
     */
    protected function getEasyCreditParameters()
    {
        $easyCreditParameters = [];

        if ($this->isEasyCreditPayment()) {
            $easyCreditParameters['customParameters']['RISK_ANZAHLBESTELLUNGEN'] =
                $this->customer->getCustomerOrderCount();
            $easyCreditParameters['customParameters']['RISK_BESTELLUNGERFOLGTUEBERLOGIN'] =
                $this->customer->isLoggedIn() ? 'true' : 'false';
            $easyCreditParameters['customParameters']['RISK_KUNDENSTATUS'] =
                $this->customer->getCustomerStatus();
            $easyCreditParameters['customParameters']['RISK_KUNDESEIT'] =
                $this->customer->getCustomerCreatedDate();
        }

        return $easyCreditParameters;
    }

    protected function isSubscriptionOrder()
    {

        $quote = $this->getCheckoutSession()->getQuote();
        $items = $quote->getAllItems();
        $isSubscriptionProduct = false;

        if (count($items) > 0) {
            $item = $items[0];

            $subscription_plan = $item->getOptionByCode('subscription_plan');
            if (!empty($subscription_plan)) {
                $isSubscriptionProduct = true;
            }
        }

        return $isSubscriptionProduct;
    }

    /**
     * get cart items parameters
     * @return array
     */
    protected function getCartItemsParameters()
    {
        $cartItemsParameters = [];

        if ($this->isKlarnaPayment() || $this->isEasyCreditPayment()) {
            $cartItems = [];
            $count = 0;
            $qty = 1;
            $itemId = '';
            $orderAllItems = $this->order->getAllItems();

            foreach ($orderAllItems as $orderItem) {
                $product = $orderItem->getProduct();

                if ($product->getTypeId() != \Magento\Catalog\Model\Product\Type::TYPE_SIMPLE) {
                    $qty = $orderItem->getQty();
                    $itemId = $orderItem->getItemId();
                    continue;
                }

                $finalPrice = (float)$product->getFinalPrice();
                $discountAmount = (float)$orderItem->getDiscountAmount();
                $taxAmount = (float)$orderItem->getTaxAmount();
                $price = (float)$product->getPrice();
                $priceIncludeTax = $price + $taxAmount;

                if ($this->isKlarnaPayment()) {
                    $quantity = (int)$orderItem->getQtyOrdered();
                    $cartItems[$count]['discount'] = ($price - $finalPrice) / $price * 100;
                    $cartItems[$count]['tax'] = $orderItem->getTaxPercent();
                } else {
                    $parentItemId = $orderItem->getParentItemId();

                    if ($parentItemId == $itemId) {
                        $quantity = $orderItem->getQty() * $qty;
                    } else {
                        $quantity = $orderItem->getQty();
                    }
                }

                $cartItems[$count]['merchantItemId'] = (int)$product->getId();
                $cartItems[$count]['quantity'] = $quantity;
                $cartItems[$count]['name'] = $orderItem->getName();
                $cartItems[$count]['price'] = $priceIncludeTax;

                $count++;
            }

            $cartItemsParameters['cartItems'] = $cartItems;
        }

        return $cartItemsParameters;
    }

    /**
     * get paydirekt parameters
     * @return array
     */
    protected function getPaydirektParameters()
    {
        $paydirektParameters = [];

        if ($this->paymentMethod->getCode() == 'viveum_paydirekt') {
            $paydirektParameters['customParameters']['paydirektMinimumAge'] =
                $this->paymentMethod->getConfigData('minimum_age');
            $paydirektParameters['customParameters']['paydirektPaymentIsPartial'] =
                $this->paymentMethod->paydirektPaymentIsPartial();
        }

        return $paydirektParameters;
    }

    /**
     * get Credit cards 3D parameters
     * @return array
     */
    protected function getCC3DParameters()
    {
        $cc3DParameters = [];

        if ($this->paymentMethod->getCode() == 'viveum_ccsaved' || $this->isSubscriptionOrder()) {
            $cc3DParameters['3D']['amount'] = $this->order->getGrandTotal();
            $cc3DParameters['3D']['currency'] = $this->getOrderCurrency();;
        }

        return $cc3DParameters;
    }

    /**
     * get custom parameters
     * @return array
     */
    public function getCustomParameters()
    {
        $customParameters = [];

        $customParameters['customParameters']['SHOP_VERSION'] = $this->paymentMethod->getShopVersion();
        $customParameters['customParameters']['PLUGIN_VERSION'] = $this->paymentMethod->getPluginVersion();

        return $customParameters;
    }


    /**
     * get paypal saved parameters
     * @param  boolean $registrationId
     * @param  boolean $transactionId
     * @return array
     */
    protected function getPaypalSavedParameters($registrationId = false, $transactionId = false)
    {
        $paypalSavedParameters = [];

        if ($this->isSubscriptionOrder()) {

            $paymentParameters = $this->getSubscriptionRegisterParameters();
            $this->logger->info("getPaypalSavedParameters paymentParameters", $paymentParameters);

            $paypalSavedParameters = array_merge(
                $this->paymentMethod->getCredentials(),
                $this->getTransactionParameters(),
                $paymentParameters,
            );
        } else {
            $paypalSavedParameters = array_merge(
                $this->paymentMethod->getCredentials(),
                $this->getTransactionParameters(),
                $this->getCustomParameters()
            );
        }

        $paypalSavedParameters['paymentType'] = $this->paymentMethod->getPaymentType();
        $paypalSavedParameters['standingInstruction.mode'] = 'REPEATED';
        $paypalSavedParameters['standingInstruction.source'] = 'MIT';
        $paypalSavedParameters['standingInstruction.type'] = 'UNSCHEDULED';

        if ($registrationId) {
            $paypalSavedParameters['transactionId'] = $transactionId;
        }

        return $paypalSavedParameters;
    }

    /**
     * check if the PayPal recurring is used then the payment type is not sent
     * @return boolean
     */
    protected function isSendPaymentTypeParameter()
    {
        if (
            $this->paymentMethod->getCode() == 'viveum_paypalsaved'
            && $this->order->getPayment()->getViveum_user_consent() == 1
            && !$this->paymentMethod->isAdminLoggedIn()
        ) {
            return false;
        }
        return true;
    }

    /**
     * get subscription register parameters
     * @return array
     */
    protected function getSubscriptionRegisterParameters($transactionId = false)
    {
        $subscriptionRegisterParameters = [];
        $subscriptionRegisterParameters = array_merge(
            $this->paymentMethod->getCredentials(),
            $this->getCustomerInformationByOrder(),
            $this->getCustomerAddressByOrder(),
            $this->getCartItemsParameters(),
            $this->getCC3DParameters(),
            $this->getCustomParameters(),
            $this->getTransactionParameters(),
            $this->getRegistrationParameters()
        );

        // $subscriptionRegisterParameters['createRegistration'] = 'true';
        $subscriptionRegisterParameters['paymentType'] = $this->paymentMethod->getPaymentType();

        if ($this->paymentMethod->getCode() == 'viveum_paypalsaved' && $this->order->getPayment()->getViveum_user_consent() == 1) {
            unset($subscriptionRegisterParameters['paymentType']);
        }

        $subscriptionRegisterParameters['customParameters']['amount'] = $subscriptionRegisterParameters['amount'];
        $subscriptionRegisterParameters['customParameters']['currency'] = $subscriptionRegisterParameters['currency'];
        $subscriptionRegisterParameters['customParameters']['isSubscription'] = 'true';

        $planId = $this->getPlanIdFromQuote($this->getQuote());
        $quote = $this->getQuote();

        $subscriptionRegisterParameters['customParameters']['planId'] = $planId;
        $subscriptionRegisterParameters['customParameters']['planCharge'] = $subscriptionRegisterParameters['amount'];
        $subscriptionRegisterParameters['customParameters']['quoteId'] = $quote->getId();
        $subscriptionRegisterParameters['customParameters']['customerId'] = $quote->getCustomerId() ?? 'NULL';
        $subscriptionRegisterParameters['customParameters']['customerEmail'] = $quote->getCustomerEmail();
        $subscriptionRegisterParameters['customParameters']['customerFirstname'] = $quote->getCustomerFirstname();
        $subscriptionRegisterParameters['customParameters']['customerLastname'] = $quote->getCustomerLastname();

        return $subscriptionRegisterParameters;
    }

    /**
     * get recurring parameters
     * @param  boolean|string $transactionId
     * @return array
     */
    protected function getRecurringParameters($transactionId = false)
    {
        $recurringParameters = [];
        $recurringParameters = array_merge(
            $this->paymentMethod->getCredentials(),
            $this->customer->getCustomerInformation(),
            $this->customer->getDefaultBillingAddress(),
            $this->getCustomParameters()
        );
        $recurringParameters['amount'] = $this->paymentMethod->getRegisterAmount();
        $recurringParameters['currency'] = $this->getRecurringCurrency();

        if ($this->paymentMethod->getCode() != 'viveum_paypalsaved') {
            $recurringParameters['paymentType'] = $this->paymentMethod->getPaymentType();
        }
        $recurringParameters['transactionId'] = $this->getRecurringParametersTransactionId($transactionId);
        $recurringParameters['standingInstruction.mode'] = 'INITIAL';
        $recurringParameters['standingInstruction.source'] = 'CIT';
        $recurringParameters['standingInstruction.type'] = 'RECURRING';
        $recurringParameters['standingInstruction.recurringType'] = 'SUBSCRIPTION';
        $recurringParameters['createRegistration'] = 'true';

        return $recurringParameters;
    }


    /**
     * get a transaction id of payment recurring
     * @param  boolean $transactionId
     * @return string
     */
    protected function getRecurringParametersTransactionId($transactionId = false)
    {
        if ($transactionId) {
            return $transactionId;
        } else {
            return $this->customer->getId();
        }
    }

    /**
     * get a recurring currency
     * @return [type]
     */
    protected function getRecurringCurrency()
    {
        $storeManager = $this->_objectManager->create('Magento\Store\Model\StoreManagerInterface');
        $store = $storeManager->getStore();
        return $store->getCurrentCurrencyCode();
    }

    /**
     * redirect to the payment error page
     * @param  string  $errorIdentifier
     * @param  boolean|string $transactionId
     * @param  string  $url
     * @return void
     */
    public function redirectError($errorIdentifier, $transactionId = false, $url = 'checkout/cart')
    {
        if ($this->order instanceof \Magento\Sales\Model\Order) {
            $this->order->cancel()->save();
        }
        $errorMessage = __($errorIdentifier);
        if ($transactionId) {
            $errorMessage .= ' (' . __('BACKEND_TT_TRANSACTION_ID') . ' : ' . $transactionId . ')';
        }
        $this->messageManager->addError($errorMessage);
        $this->_redirect($url, ['_secure' => true]);
    }

    /**
     * redirect to the payment error page at my payment information
     * @param  string $generalError
     * @param  string $detailError
     * @param  string $informationId
     * @param  string $url
     * @return string
     */
    protected function redirectErrorRecurring(
        $generalError,
        $detailError = null,
        $informationId = null,
        $url = 'viveum/payment/information'
    ) {
        if ($generalError) {
            $errorMessage = __($generalError);
        } elseif ($informationId) {
            $errorMessage = __('ERROR_MC_UPDATE');
        } else {
            $errorMessage = __('ERROR_MC_ADD');
        }
        if ($detailError) {
            $errorMessage .= ' : ' . __($detailError);
        }
        $this->messageManager->addError($errorMessage);
        $this->_redirect($url, ['_secure' => true]);
    }

    /**
     * redirect to the register or change success page
     * @param  string $successIdentifier
     * @param  string $url
     * @return string
     */
    protected function redirectSuccessRecurring($successIdentifier, $url = 'viveum/payment/information')
    {
        $this->messageManager->addSuccess(__($successIdentifier));
        $this->_redirect($url, ['_secure' => true]);
    }

    /**
     * get a checkout session
     * @return object
     */
    protected function getCheckoutSession()
    {
        return $this->checkoutSession;
    }

    /**
     * get a Quote
     * @return object
     */
    protected function getQuote()
    {
        if (!$this->quote) {
            $this->quote = $this->getCheckoutSession()->getQuote();
        }
        return $this->quote;
    }

    protected function getPlanIdFromQuote($quote)
    {
        $quote = $this->getCheckoutSession()->getQuote();
        $items = $quote->getAllItems();
        $planId = null;

        if (count($items) > 0) {
            $item = $items[0];

            $subscription_plan = $item->getOptionByCode('subscription_plan');
            if (!empty($subscription_plan)) {
                $planId = $subscription_plan->getValue();
            }
        }

        return $planId;
    }
    /**
     * get an order
     * @return object
     */
    protected function getOrder()
    {
        $order = $this->_objectManager->create('Magento\Sales\Model\Order');
        $order->load($this->checkoutSession->getLastOrderId());

        return $order;
    }

    /**
     * get an order based on increment id
     * @param  int $incrementId
     * @return object
     */
    public function getOrderByIncerementId($incrementId)
    {
        $order = $this->_objectManager->create('Magento\Sales\Model\Order');
        $order->loadByIncrementId($incrementId);

        return $order;
    }

    /**
     * get an order based on id
     * @param  int $id
     * @return object
     */
    protected function getOrderById($id)
    {
        $order = $this->_objectManager->create('Magento\Sales\Model\Order');
        $order->load($id);

        return $order;
    }

    /**
     * get information parameters
     * @return array
     */
    public function getInformationParamaters()
    {
        $informationParameters = [];
        $informationParameters['customerId'] = $this->customer->getId();
        $informationParameters['serverMode'] = $this->paymentMethod->getServerMode();
        $informationParameters['channelId'] = $this->paymentMethod->getChannelId();
        $informationParameters['paymentGroup'] =  $this->paymentMethod->getPaymentGroup();

        return $informationParameters;
    }

    /**
     * create an Invoice
     * @return void
     */
    public function createInvoice()
    {
        /* var Magento\Sales\Model\Service\InvoiceService $invoiceService */
        $invoiceService = $this->_objectManager->create('Magento\Sales\Model\Service\InvoiceService');

        $invoice = $invoiceService->prepareInvoice($this->order);
        $invoice->setRequestedCaptureCase(\Magento\Sales\Model\Order\Invoice::CAPTURE_ONLINE);
        $invoice->register();
        $invoice->getOrder()->setCustomerNoteNotify(false);
        $invoice->getOrder()->setIsInProcess(true);

        $transactionSave = $this->_objectManager->create('Magento\Framework\DB\Transaction');
        $transactionSave->addObject($invoice)->addObject($invoice->getOrder())->save();

        $invoiceSender = $this->_objectManager->create('Magento\Sales\Model\Order\Email\Sender\InvoiceSender');
        $invoiceSender->send($invoice);
    }

    /**
     * save an order additional information
     * @param  array $paymentStatus
     * @return void
     */
    public function saveOrderAdditionalInformation($paymentStatus)
    {
        $payment = $this->order->getPayment();
        $paymentMethod = $this->paymentMethod->getCode();

        $this->setDDAdditionalInformation($paymentMethod, $payment);
        $this->setEasyCreditAdditionalInformation($paymentMethod, $payment, $paymentStatus);

        if (isset($paymentStatus['merchantTransactionId'])) {
            $payment->setAdditionalInformation('TRANSACTION_ID', $paymentStatus['merchantTransactionId']);
        }
        if (isset($paymentStatus['descriptor'])) {
            $payment->setAdditionalInformation('SHORT_ID', $this->getShortIdFromDescriptor($paymentStatus['descriptor']));
        }
        if (isset($paymentStatus['id'])) {
            $payment->setAdditionalInformation('REFERENCE_ID', $paymentStatus['id']);
        }
        if (isset($paymentStatus['currency'])) {
            $payment->setAdditionalInformation('CURRENCY', $paymentStatus['currency']);
        }
        if (isset($paymentStatus['amount'])) {
            $payment->setAdditionalInformation('AMOUNT', $paymentStatus['amount']);
        }
        if (isset($paymentStatus['card']['last4Digits'])) {
            $payment->setCcLast4($paymentStatus['card']['last4Digits']);
        }
        if (isset($paymentStatus['card']['expiryMonth'])) {
            $payment->setCcSsStartMonth($paymentStatus['card']['expiryMonth']);
        }
        if (isset($paymentStatus['card']['expiryYear'])) {
            $payment->setCcExpYear($paymentStatus['card']['expiryYear']);
        }
        if (isset($paymentStatus['paymentType'])) {
            $payment->setAdditionalInformation('PAYMENT_TYPE', $paymentStatus['paymentType']);
        }
        if (isset($paymentStatus['result']['code'])) {
            $payment->setAdditionalInformation('RESULT_CODE', $paymentStatus['result']['code']);
        }
        if (isset($paymentStatus['result']['description'])) {
            $payment->setAdditionalInformation('RESULT_DESCRIPTION', $paymentStatus['result']['description']);
        }
        if (isset($paymentStatus['risk']['score'])) {
            $payment->setAdditionalInformation('RISK_SCORE', $paymentStatus['risk']['score']);
        }
        if (isset($paymentStatus['paymentType']) && isset($paymentStatus['result']['code'])) {
            $orderStatusCode = $this->setOrderStatusCode($paymentStatus);
            $payment->setAdditionalInformation('ORDER_STATUS_CODE', $orderStatusCode);
        }

        $this->order->save();
    }

    private function getShortIdFromDescriptor($descriptor)
    {
        $descriptors = explode(' ', $descriptor);
        $short_id    = $descriptors[0];

        return $short_id;
    }

    /**
     * set the Direct Debit Additional Information
     * @param string $paymentMethod
     * @param object $payment
     */
    public function setDDAdditionalInformation($paymentMethod, $payment)
    {
        if ($paymentMethod == 'viveum_directdebit' || $paymentMethod == 'viveum_ddsaved') {
            if (!$this->paymentMethod->isAdminLoggedIn()) {
                $payment->setAdditionalInformation('MANDATE_ID', $this->checkoutSession->getMandateId());
                $payment->setAdditionalInformation('MANDATE_DATE', $this->checkoutSession->getMandateDate());
            }
        }
    }

    /**
     * set the Direct Debit Additional Information
     * @param string $paymentMethod
     * @param object $payment
     * @param array $paymentStatus
     */
    public function setEasyCreditAdditionalInformation($paymentMethod, $payment, $paymentStatus)
    {
        if ($paymentMethod == 'viveum_easycredit') {
            if (isset($paymentStatus['resultDetails']['tilgungsplanText'])) {
                $payment->setAdditionalInformation(
                    'redemption_plan',
                    $paymentStatus['resultDetails']['tilgungsplanText']
                );
            }
            if (isset($paymentStatus['resultDetails']['vorvertraglicheInformationen'])) {
                $payment->setAdditionalInformation(
                    'pre_contract_information_url',
                    $paymentStatus['resultDetails']['vorvertraglicheInformationen']
                );
            }
            if (isset($paymentStatus['resultDetails']['ratenplan.zinsen.anfallendeZinsen'])) {
                $payment->setAdditionalInformation(
                    'easycredit_sum_of_interest',
                    $paymentStatus['resultDetails']['ratenplan.zinsen.anfallendeZinsen']
                );
            }
            if (isset($paymentStatus['resultDetails']['ratenplan.gesamtsumme'])) {
                $payment->setAdditionalInformation(
                    'easycredit_order_total',
                    $paymentStatus['resultDetails']['ratenplan.gesamtsumme']
                );
            }
            if (isset($paymentStatus['customParameters']['orderId'])) {
                $payment->setAdditionalInformation(
                    'easycredit_order_id',
                    $paymentStatus['customParameters']['orderId']
                );
            }
        }
    }

    /**
     * set an order status code
     * @param array $paymentStatus
     */
    protected function setOrderStatusCode($paymentStatus)
    {
        $isInReview = $this->helperPayment->isSuccessReview($paymentStatus['result']['code']);

        if ($isInReview) {
            return 'IR';
        } else {
            return $paymentStatus['paymentType'];
        }
    }

    /**
     * redirect to error page when update a payment status
     * @param  string  $errorMessage
     * @param  string  $orderId
     * @param  boolean|string $detailError
     * @param  string  $url
     * @return void
     */
    public function redirectErrorOrderDetail(
        $errorMessage,
        $orderId,
        $detailError = false,
        $url = 'sales/order/view'
    ) {
        $this->messageManager->addError(__($errorMessage, $orderId, $detailError));
        $this->_redirect($url, ['order_id' => (int)$orderId]);
    }

    /**
     * redirect to success page when update a payment status
     * @param  string $successIdentifier
     * @param  string $orderId
     * @param  string $url
     * @return void
     */
    public function redirectSuccessOrderDetail($successIdentifier, $orderId, $url = 'sales/order/view')
    {
        $this->messageManager->addSuccess(__($successIdentifier, $orderId));
        $this->_redirect($url, ['order_id' => (int)$orderId]);
    }

    /**
     * get error details from easycredit
     *
     * @param array $paymentResponse
     * @return array
     */
    public function getEasyCreditErrorDetail($paymentResponse)
    {
        $errorResults = $this->explodeByMultiDelimiter(
            ["{", "}"],
            $paymentResponse['resultDetails']['Error']
        );
        $errorResults = explode(", ", $errorResults[1]);
        foreach ($errorResults as $errorResult) {
            $errorResultValue = explode("=", $errorResult);
            $easyCreditErrorDetail[$errorResultValue[0]] =
                sizeof($errorResultValue) > 1 ? trim($errorResultValue[1], "'") : trim($errorResultValue[0], "'");
        }
        return $easyCreditErrorDetail;
    }

    /**
     * explode string with multi delimiter
     *
     * @param array $delimiters
     * @param string $string
     * @return array
     */
    public function explodeByMultiDelimiter($delimiters, $string)
    {
        $string = str_replace($delimiters, $delimiters[0], $string);
        $explodedString = explode($delimiters[0], $string);
        return $explodedString;
    }

    public function emptyCart()
    {
        $this->logger->info("emptyCart started");
        // $objectManager = \Magento\Framework\App\ObjectManager::getInstance();
        // $cartObject = $objectManager->create('Magento\Checkout\Model\Cart');

        // $cartObject->getQuote()->setTotalsCollectedFlag(false);
        // $cartObject->save();

        $objectManager = \Magento\Framework\App\ObjectManager::getInstance();
        $cartObject = $objectManager->create('Magento\Checkout\Model\Cart')->truncate();
        $cartObject->saveQuote();
    }
}
