<?php

namespace Viveum\Viveum\Controller\Payment;

use Magento\Framework\DataObject;


class Response extends \Viveum\Viveum\Controller\Payment
{
    private $isNotRedirect;
    protected $logger;

    /**
     * @var \Magento\Store\Model\StoreManagerInterface
     */
    private $_storeManagement;

    /**
     * @var \Magento\Framework\App\ObjectManager
     */
    private $_objectManagement;

    /**
     * @var \Magento\Quote\Model\QuoteRepository
     */
    private $_quoteRepository;

    private $_quoteFactory;

    /**
     * @var \Magento\Quote\Model\QuoteManagement
     */
    private $_quoteManagement;

    protected $_customerRepositoryInterface;

    public function __construct(
        \Magento\Framework\App\Action\Context $context,
        \Magento\Checkout\Model\Session $checkoutSession,
        \Magento\Framework\Locale\ResolverInterface $localeResolver,
        \Magento\Framework\View\Result\PageFactory $resultPageFactory,
        \Viveum\Viveum\Helper\Payment $helperPayment,
        \Viveum\Viveum\Helper\Curl $curl,
        \Viveum\Viveum\Model\Payment\Information $information,
        \Viveum\Viveum\Model\Customer\Customer $customer,
        \Viveum\Viveum\Model\Subscription\Subscription $subscription,
        \Viveum\Viveum\Model\Subscription\SubscriptionPayment $subscriptionPayment,
        \Magento\Customer\Api\CustomerRepositoryInterface $customerRepositoryInterface
    ) {
        parent::__construct(
            $context,
            $checkoutSession,
            $localeResolver,
            $resultPageFactory,
            $helperPayment,
            $curl,
            $information,
            $customer,
            $subscription,
            $subscriptionPayment
        );
        $this->logger = new \Viveum\Viveum\Helper\Logger("Payment\Response");

        $this->_customerRepositoryInterface = $customerRepositoryInterface;
        $this->_objectManagement = \Magento\Framework\App\ObjectManager::getInstance();
        $this->_storeManagement = $this->_objectManagement->get(\Magento\Store\Model\StoreManagerInterface::class);
        $this->_quoteRepository = $this->_objectManagement->get(\Magento\Quote\Model\QuoteRepository::class);
        $this->_quoteManagement = $this->_objectManagement->get(\Magento\Quote\Model\QuoteManagement::class);
        $this->_quoteFactory = new  \Magento\Quote\Model\QuoteFactory($this->_objectManagement);
    }
    /**
     * get payment responses
     * @return void
     */
    public function execute()
    {
        $this->setPaymentMethodByCode($this->getRequest()->getParam('payment_method'));

        $repeatedPaypal = $this->getRequest()->getParam('repeated_paypal');
        if ($repeatedPaypal) {
            $orderId = $this->getRequest()->getParam('orderId');
            $this->order = $this->getOrderByIncerementId($orderId);
            $this->processDebitPaypalSaved();
        } else {
            $isServerToServer = $this->getRequest()->getParam('server_to_server');
            $this->processPayment($isServerToServer);
        }
    }

    public function setNotRedirected($isNotRedirect)
    {
        $this->isNotRedirect = $isNotRedirect;
    }

    /**
     * Set payment method object by code
     * @param string $code
     * @return void
     */
    public function setPaymentMethodByCode($code)
    {
        $this->logger->info("setPaymentMethodByCode code", $code);
        $this->paymentMethod = $this->createPaymentMethodObjectByPaymentMethod($code);
    }

    /**
     * process payment
     * @param  boolean $isServerToServer
     * @return void
     */
    public function processPayment($isServerToServer = false)
    {
        $this->logger->info("processPayment getParams", $this->getRequest()->getParams());

        $checkoutId = $this->getRequest()->getParam('id');
        $paymentMethod = $this->paymentMethod;
        $credentials = $paymentMethod->getCredentials();

        if ($isServerToServer == $paymentMethod::SYNCHRONOUS) {
            $paymentStatus = $this->checkoutSession->getInitialPaymentResponse();
        } else {
            $paymentStatus = $this->helperPayment->getPaymentStatus($checkoutId, $credentials, $isServerToServer);
        }

        $this->logger->info("processPayment paymentStatus", $paymentStatus);


        if (isset($paymentStatus['isValid']) && $paymentStatus['isValid']) {

            $this->processPaymentResponse($paymentStatus['response']);
        } else {
            $orderId = $this->checkoutSession->getOrderId();
            $this->order = $this->getOrderByIncerementId($orderId);
            if (isset($paymentStatus['response'])) {
                $this->redirectError($paymentStatus['response'], $this->checkoutSession->getTransactionId());
            } else {
                $this->redirectError('ERROR_GENERAL_NORESPONSE', $this->checkoutSession->getTransactionId());
            }
        }
    }

    public function processSchedulePayment($paymentResponse)
    {
        try {

            $registrationResponse = $paymentResponse;

            $paymentMethod = $this->paymentMethod;
            $credentials = $paymentMethod->getCredentials();

            $scheduleParameters = [];

            $scheduleParameters['paymentType'] = $this->paymentMethod->getPaymentType();
            $scheduleParameters['planId'] = $paymentResponse['customParameters']['planId'];
            $scheduleParameters['amount'] =  $paymentResponse['customParameters']['amount'];
            $scheduleParameters['currency'] = $paymentResponse['customParameters']['currency'];
            $scheduleParameters['registrationId'] = $paymentResponse['registrationId'];
            $scheduleParameters['merchantTransactionId'] = $paymentResponse['customParameters']['orderId'];
            $scheduleParameters['paymentMethod'] = $paymentMethod->getCode();

            if (!empty($paymentResponse['standingInstruction']['initialTransactionId'])) {
                $scheduleParameters['standingInstruction']['initialTransactionId'] = $paymentResponse['standingInstruction']['initialTransactionId'];
            }

            $scheduleParameters = array_merge($scheduleParameters, $credentials);
            unset($scheduleParameters['testMode']);

            $paymentResult = $this->helperPayment->createSchedule($scheduleParameters);
            $this->logger->info("processSchedulePayment paymentResult", $paymentResult);

            $paymentResponse = $paymentResult['response'];
            $returnCode = $paymentResponse['result']['code'];
            $returnMessage = $this->helperPayment->getErrorIdentifier($returnCode);

            $transactionResult = $this->helperPayment->getTransactionResult($returnCode);

            if ($transactionResult == 'ACK') {
                $registrationResponse['subscription_id'] = $paymentResponse['id'];
                return $this->validatePayment($paymentResponse);
            } elseif ($transactionResult == 'NOK') {
                if (!$this->isNotRedirect) {
                    $this->redirectError($returnMessage, $registrationResponse['customParameters']['orderId']);
                } else {
                    $this->logger->info("processSchedulePayment Transaction result is NOK", "Order will be canceled");
                    $this->order->cancel()->save();
                    return "Transaction result is NOK";
                }
            } elseif ($transactionResult == 'PD' && $paymentResult['paymentBrand'] == "SOFORTUEBERWEISUNG") {
                $this->order->cancel()->save();
            } else {
                if (!$this->isNotRedirect) {
                    $this->redirectError('ERROR_UNKNOWN', $registrationResponse['customParameters']['orderId']);
                } else {
                    $this->logger->info("processSchedulePayment Transaction result is UNKNOWN", "Order will be canceled");
                    $this->order->cancel()->save();
                    return "Transaction result is UNKNOWN";
                }
            }
        } catch (\Exception $e) {
            if (!$this->isNotRedirect) {
                $this->redirectError('ERROR_UNKNOWN', $registrationResponse['customParameters']['orderId']);
            } else {
                $this->logger->info("processSchedulePayment Transaction result is UNKNOWN", "Order will be canceled");
                $this->order->cancel()->save();
                return "Transaction result is UNKNOWN";
            }
        }
    }

    public function descheduleIfFailed($subscriptionId)
    {
        $this->paymentMethod = $this->createPaymentMethodObjectByPaymentMethod('ccsaved');

        $deleteParameters = $this->paymentMethod->getCredentials();
        $deleteStatus = $this->helperPayment->deSchedule($subscriptionId, $deleteParameters);

        if ($deleteStatus['isValid']) {
            $returnCode = $deleteStatus['response']['result']['code'];
            $transactionResult = $this->helperPayment->getTransactionResult($returnCode);

            if ($transactionResult == 'ACK') {
                return true;
            } else {
                return false;
            }
        } else {
            return false;
        }
    }

    public function processSubscriptionFromWebhook($payload)
    {
        $this->logger->info("processSubscriptionFromWebhook payload", $payload);

        $returnCode = $payload['result']['code'];
        $transactionResult = $this->helperPayment->getTransactionResult($returnCode);

        $paymentId = $payload['id'];

        if ($payload['source'] == 'SCHEDULER') {
            $subscription = $this->subscription->getByTransactionId($payload['merchantTransactionId']);
            $this->logger->info("processSubscriptionFromWebhookcheck subscription", $subscription);

            if (count($subscription) > 0) {
                $subscriptionId = $subscription[0]['subscription_id'];
                $subscription = $subscription[0];
            } else {
                return false;
            }
        } else {
            if ($payload['customParameters']['paymentMethod'] == 'viveum_ccsaved') return true;
            $subscriptionId = $payload['customParameters']['subscriptionId'];
        }

        $parameters = [];
        $parameters['from_webhook'] = 1;
        $parameters['subscription_id'] = $subscriptionId;
        $parameters['payment_id'] = $paymentId;
        $parameters['amount'] = $payload['currency'] . ' ' . $payload['amount'];
        $parameters['transaction_id'] = $payload['merchantTransactionId'];

        if ($transactionResult == 'ACK') {
            $parameters['status'] = 'success';

            if ($payload['source'] == 'SCHEDULER') {
                $this->createOrderSubscriptionFromWebhook($subscription, $payload);
            }
        } else {
            $parameters['status'] = 'failed';
        }

        $this->logger->info("processSubscriptionFromWebhook parameters to save", $parameters);

        $this->subscriptionPayment->insertPayment($parameters);
        return true;
    }

    public function processPaymentResponse($paymentResponse)
    {
        $this->logger->info("processPaymentResponse paymentResponse", $paymentResponse);

        $returnCode = $paymentResponse['result']['code'];
        $returnMessage = $this->helperPayment->getErrorIdentifier($returnCode);
        $transactionResult = $this->helperPayment->getTransactionResult($returnCode);

        $this->logger->info("processPaymentResponse returnMessage", $returnMessage);
        $this->logger->info("processPaymentResponse transactionResult", $transactionResult);

        if ($this->isEasyCreditPayment()) {
            $this->quote = $this->getQuote();
            $this->order = $this->quote;
            if (isset($paymentResponse['resultDetails']['Error'])) {
                $returnMessage = $this->getEasyCreditErrorDetail($paymentResponse);
                if (isset($returnMessage['field']) && $returnMessage['field'] !== 'null') {
                    $easyCreditErrorDetail = $returnMessage['field'] . ': ' . $returnMessage['renderedMessage'];
                } else {
                    $easyCreditErrorDetail = $returnMessage['renderedMessage'];
                }

                $this->logger->info("processPaymentResponse easyCreditErrorDetail", $easyCreditErrorDetail);

                $this->redirectError($easyCreditErrorDetail, $this->checkoutSession->getTransactionId());
                return false;
            }
        } else {
            if (!isset($paymentResponse['customParameters']['orderId'])) {
                $orderId = $this->checkoutSession->getOrderId();
                $this->order = $this->getOrderByIncerementId($orderId);
                $this->redirectError($returnMessage, $this->checkoutSession->getTransactionId());
                return false;
            }
            $orderId = $paymentResponse['customParameters']['orderId'];
            $this->order = $this->getOrderByIncerementId($orderId);

            $this->logger->info("processPaymentResponse orderId", $orderId);
        }

        if (!isset($paymentResponse['customParameters']['isSubscription']) || $this->paymentMethod->getCode() == 'viveum_ccsaved') {
            $this->saveOrderAdditionalInformation($paymentResponse);
        }

        if ($transactionResult == 'ACK') {
            if ($this->isEasyCreditPayment()) {
                $this->_redirect('viveum/payment/review', ['_secure' => true]);
            } else {
                if (isset($paymentResponse['customParameters']['isSubscription'])) {
                    if ($this->paymentMethod->getCode() != 'viveum_paypalsaved') {
                        $this->processSchedule($paymentResponse);
                    }
                }

                return $this->validatePayment($paymentResponse);
            }
        } elseif ($transactionResult == 'NOK') {
            if (!$this->isNotRedirect) {
                $this->redirectError($returnMessage, $paymentResponse['merchantTransactionId']);
            } else {
                $this->logger->info("processPaymentResponse Transaction result is NOK", "Order will be canceled");
                $this->order->cancel()->save();
                return "Transaction result is NOK";
            }
        } elseif ($transactionResult == 'PD' && $paymentResponse['paymentBrand'] == "SOFORTUEBERWEISUNG") {
            $this->order->cancel()->save();
        } else {
            if (!$this->isNotRedirect) {
                $this->redirectError('ERROR_UNKNOWN', $paymentResponse['merchantTransactionId']);
            } else {
                $this->logger->info("processPaymentResponse Transaction result is UNKNOWN", "Order will be canceled");
                $this->order->cancel()->save();
                return "Transaction result is UNKNOWN";
            }
        }
    }

    /**
     * validate a payment
     * @param  array $paymentStatus
     * @return void
     */
    public function validatePayment($paymentStatus)
    {
        $isRecurringPayment = $this->paymentMethod->isRecurringPayment();
        if ($isRecurringPayment) {
            $this->processRecurringPayment($paymentStatus);
        }

        return $this->processSuccessPayment($paymentStatus);
    }

    /**
     * process a success payment
     * @param  array $paymentStatus
     * @return void
     */
    public function processSuccessPayment($paymentStatus)
    {
        $orderSender = $this->_objectManager->create('Magento\Sales\Model\Order\Email\Sender\OrderSender');
        $orderSender->send($this->order);

        $orderStatus = $this->setOrderStatus($paymentStatus);
        if ($orderStatus) {
            $this->order->setState('new')->setStatus($orderStatus)->save();
            $this->order->addStatusToHistory($orderStatus, '', true)->save();
        } else {
            $this->createInvoice();
        }

        if (!$this->isNotRedirect) {

            // prepare session to success or cancellation page
            $this->checkoutSession->clearHelperData();

            // "last successful quote"
            $quoteId = $this->getQuote()->getId();
            $this->checkoutSession->setLastQuoteId($quoteId)->setLastSuccessQuoteId($quoteId);

            $this->checkoutSession->setLastOrderId($this->order->getId())
                ->setLastRealOrderId($this->order->getIncrementId())
                ->setLastOrderStatus($this->order->getStatus());

            $this->deActiveQuote();

            $this->_redirect('checkout/onepage/success');
        } else {
            $msg = "Order is processed";
            $this->logger->info("processSuccessPayment", $msg);
            return $msg;
        }
    }

    /**
     * set an order status
     * @param array $paymentStatus
     */
    public function setOrderStatus($paymentStatus)
    {
        $isInReview = $this->helperPayment->isSuccessReview($paymentStatus['result']['code']);

        if ($isInReview) {
            $orderStatus = \Magento\Sales\Model\Order::STATE_PAYMENT_REVIEW;
        } else {
            $orderStatus = false;
            if (isset($paymentStatus['paymentType']) && $paymentStatus['paymentType'] == 'PA') {
                $orderStatus = \Viveum\Viveum\Model\Method\AbstractMethod::STATUS_PA;
            }
        }
        return $orderStatus;
    }

    /**
     * deactive quote
     * @return void
     */
    protected function deActiveQuote()
    {
        $quote = $this->_objectManager->create('Magento\Quote\Model\Quote');
        $quote->loadActive($this->checkoutSession->getLastQuoteId());
        $quote->setReservedOrderId($this->order->getIncrementId());
        $quote->setIsActive(false)->save();
    }

    /**
     * process a payment recurring
     * @param  array &$paymentStatus
     * @return void
     */
    public function processRecurringPayment(&$paymentStatus)
    {
        $this->logger->info("processRecurringPayment paymentStatus", $paymentStatus);

        if (
            $this->paymentMethod->getCode() == 'viveum_paypalsaved' &&
            $this->order->getPayment()->getViveum_user_consent() == 1
        ) {
            $paymentStatus['registrationId'] = $paymentStatus['id'];
            unset($paymentStatus['id']);
            $registrationId = $paymentStatus['registrationId'];
            $paymentAccount = $paymentStatus['virtualAccount'];
            $paymentBrand = $paymentStatus['paymentBrand'];

            $debitPaypalSaved = $this->processDebitPaypalSaved($registrationId, $paymentStatus['merchantTransactionId']);

            if ($debitPaypalSaved) {
                $paymentStatus = $debitPaypalSaved;
                $paymentStatus['registrationId'] = $registrationId;
                $paymentStatus['virtualAccount'] = $paymentAccount;
                $paymentStatus['paymentBrand'] = $paymentBrand;
            }
        }

        if (isset($paymentStatus['registrationId'])) {
            $informationParamaters = $this->getInformationParamaters();
            $this->logger->info("processRecurringPayment informationParamaters", $informationParamaters);

            $isRegistrationExist =
                $this->information->isRegistrationExist($informationParamaters, $paymentStatus['registrationId']);

            if (!$isRegistrationExist) {
                $registrationParameters = array_merge(
                    $informationParamaters,
                    $paymentStatus,
                    $this->paymentMethod->getAccount($paymentStatus)
                );

                $this->logger->info("processRecurringPayment registrationParameters", $registrationParameters);

                if (!empty($registrationParameters['customerId'])) {
                    $this->information->insertRegistration($registrationParameters);
                }
            }
        }
    }

    public function setNumberFormat($number)
    {
        if (empty($number)) $number = '';
        $number = (float) str_replace(',', '.', $number);
        return number_format($number, 2, '.', '');
    }

    /**
     * process a debit paypal saved
     * @param  boolean|string $registrationId
     * @param  boolean|string $transactionId
     * @return void
     */
    public function processDebitPaypalSaved($registrationId = false, $transactionId = false)
    {
        if ($registrationId) {
            $referenceId = $registrationId;
        } else {
            $referenceId = $this->getRequest()->getParam('registration_id');
        }

        $paypalSavedParameters = $this->getPaypalSavedParameters($registrationId, $transactionId);

        if (isset($paypalSavedParameters['customParameters']['planId'])) $isSubscription = true;
        if (isset($isSubscription)) {

            $scheduleParameters = [];

            $amount = $this->setNumberFormat($paypalSavedParameters['customParameters']['amount']);
            $scheduleParameters['planId'] = $paypalSavedParameters['customParameters']['planId'];
            $scheduleParameters['paymentType'] = $this->paymentMethod->getPaymentType();
            $scheduleParameters['amount'] =  $amount;
            $scheduleParameters['currency'] = $paypalSavedParameters['customParameters']['currency'];
            $scheduleParameters['registrationId'] = $referenceId;
            $scheduleParameters['merchantTransactionId'] = $paypalSavedParameters['customParameters']['orderId'];
            $scheduleParameters['paymentMethod'] = $this->paymentMethod->getCode();

            $credentials = $this->paymentMethod->getCredentials();
            $this->logger->info("paypalSchedule credentials", $credentials);

            $scheduleParameters = array_merge($scheduleParameters, $credentials);
            unset($scheduleParameters['testMode']);

            $this->logger->info("paypalSchedule ParameterSD", $scheduleParameters);
            $paymentResult = $this->helperPayment->createSchedule($scheduleParameters);
            $this->logger->info("paypalSchedule createSD", $paymentResult);

            $paymentResponse = $paymentResult['response'];
            $subscriptionId = $paymentResponse['id'];
            $returnCode = $paymentResponse['result']['code'];
            $returnMessage = $this->helperPayment->getErrorIdentifier($returnCode);

            $transactionResult = $this->helperPayment->getTransactionResult($returnCode);

            if ($transactionResult != 'ACK') {
                $returnMessage = 'ERROR_UNKNOWN';
                $this->redirectError($returnMessage);
                return;
            }
        }

        try {
            $debitPaypalSavedResponse = $this->helperPayment->isDebitPaypalGetResponse($referenceId, $paypalSavedParameters);
            if ($debitPaypalSavedResponse['isValid']) {
                $returnCode = $debitPaypalSavedResponse['response']['result']['code'];
                $transactionResult = $this->helperPayment->getTransactionResult($returnCode);
                $this->saveOrderAdditionalInformation($debitPaypalSavedResponse['response']);
                if ($transactionResult == 'ACK') {

                    if (isset($isSubscription)) {

                        $this->logger->info("-----------------------------------------");
                        $this->logger->info("processDebitPaypalSaved insertToSubscriptionTable", $debitPaypalSavedResponse);
                        $this->logger->info("-----------------------------------------");

                        $subscriptionData = [
                            'id' => $subscriptionId,
                            'registration_id' => $referenceId,
                            'transaction_id' => $debitPaypalSavedResponse['response']['merchantTransactionId'],
                            'plan_id' => $paypalSavedParameters['customParameters']['planId'],
                            'amount' => $paypalSavedParameters['customParameters']['currency'] . ' ' . $amount,
                            'quote_id' => $paypalSavedParameters['customParameters']['quoteId'],
                            'magento_user_id' => $paypalSavedParameters['customParameters']['customerId'],
                            'customer_email' => $paypalSavedParameters['customParameters']['customerEmail'],
                            'customer_firstname' => $paypalSavedParameters['customParameters']['customerFirstname'],
                            'customer_lastname' => $paypalSavedParameters['customParameters']['customerLastname'],
                            'payment_method' => $this->paymentMethod->getCode(),
                        ];

                        $subscription = $this->subscription->insertSubscription($subscriptionData);
                        $this->logger->info("processDebitPaypalSaved result", $subscription);

                        $subscriptionData['from_webhook'] = 0;
                        $subscriptionData['subscription_id'] = $subscriptionId;
                        $subscriptionData['payment_id'] = $debitPaypalSavedResponse['response']['id'];
                        $subscriptionData['status'] = 'success';
                        $subscriptionPayment = $this->subscriptionPayment->insertPayment($subscriptionData);
                    }

                    if ($registrationId) {
                        return $debitPaypalSavedResponse['response'];
                    }
                    $debitPaypalSavedResponse['response']['registrationId'] = $referenceId;
                    $this->processSuccessPayment($debitPaypalSavedResponse['response']);
                } else {
                    if ($transactionResult == 'NOK') {
                        $returnMessage = $this->helperPayment->getErrorIdentifier($returnCode);
                    } else {
                        $returnMessage = 'ERROR_UNKNOWN';
                    }

                    if (isset($isSubscription)) $this->descheduleIfFailed($subscriptionId);
                    $this->redirectError($returnMessage, $debitPaypalSavedResponse['response']['merchantTransactionId']);
                }
            } else {
                $this->order
                    ->getPayment()
                    ->setAdditionalInformation('TRANSACTION_ID', $this->checkoutSession->getTransactionId())
                    ->save();

                if (isset($isSubscription)) $this->descheduleIfFailed($subscriptionId);
                $this->redirectError($returnMessage, $debitPaypalSavedResponse['response']);
            }
        } catch (\Exception $e) {
            $returnMessage = 'ERROR_UNKNOWN';
            if (isset($isSubscription)) $this->descheduleIfFailed($subscriptionId);
            $this->redirectError($returnMessage);
        }
    }


    public function processSchedule($data)
    {
        try {
            $this->logger->info("processSchedule data", $data);

            if (isset($data['customParameters']['planId'])) $isSubscription = true;
            if (isset($isSubscription)) {
                $scheduleParameters = [];

                $amount = $this->setNumberFormat($data['customParameters']['amount']);
                $scheduleParameters['planId'] = $data['customParameters']['planId'];
                $scheduleParameters['paymentType'] = $this->paymentMethod->getPaymentType();
                $scheduleParameters['amount'] =  $amount;
                $scheduleParameters['currency'] = $data['customParameters']['currency'];
                $scheduleParameters['registrationId'] = $data['registrationId'];
                $scheduleParameters['merchantTransactionId'] = $data['customParameters']['orderId'];
                $scheduleParameters['paymentMethod'] = $this->paymentMethod->getCode();

                $savedPayment = $this->information->getRegistrationByRegistrationId($data['registrationId']);

                if (!empty($savedPayment[0]['initial_transaction_id'])) {
                    $scheduleParameters['standingInstruction']['initialTransactionId'] = $savedPayment[0]['initial_transaction_id'];
                } elseif (!empty($data['standingInstruction']['initialTransactionId'])) {
                    $scheduleParameters['standingInstruction']['initialTransactionId'] = $data['standingInstruction']['initialTransactionId'];
                }

                $credentials = $this->paymentMethod->getCredentials();
                $this->logger->info("processSchedule credentials", $credentials);

                $scheduleParameters = array_merge($scheduleParameters, $credentials);
                unset($scheduleParameters['testMode']);

                $this->logger->info("processSchedule ParameterSD", $scheduleParameters);
                $paymentResult = $this->helperPayment->createSchedule($scheduleParameters);
                $this->logger->info("processSchedule createSD", $paymentResult);

                $paymentResponse = $paymentResult['response'];
                $subscriptionId = $paymentResponse['id'];
                $returnCode = $paymentResponse['result']['code'];
                $returnMessage = $this->helperPayment->getErrorIdentifier($returnCode);

                $transactionResult = $this->helperPayment->getTransactionResult($returnCode);

                if ($transactionResult == 'ACK') {
                    $this->logger->info("-----------------------------------------");
                    $this->logger->info("processSchedule insertToSubscriptionTable", $paymentResponse);
                    $this->logger->info("-----------------------------------------");

                    $subscriptionData = [
                        'id' => $subscriptionId,
                        'registration_id' => $data['registrationId'],
                        'transaction_id' => $data['merchantTransactionId'],
                        'plan_id' => $data['customParameters']['planId'],
                        'amount' => $data['customParameters']['currency'] . ' ' . $data['customParameters']['amount'],
                        'quote_id' => $data['customParameters']['quoteId'],
                        'magento_user_id' => $data['customParameters']['customerId'],
                        'customer_email' => $data['customParameters']['customerEmail'],
                        'customer_firstname' => $data['customParameters']['customerFirstname'],
                        'customer_lastname' => $data['customParameters']['customerLastname'],
                        'payment_method' => $this->paymentMethod->getCode(),
                    ];

                    $subscription = $this->subscription->insertSubscription($subscriptionData);
                    $this->logger->info("processSchedule subscriptionData", $subscriptionData);
                    $this->logger->info("processSchedule result", $subscription);

                    $subscriptionData['from_webhook'] = 0;
                    $subscriptionData['subscription_id'] = $subscriptionId;
                    $subscriptionData['payment_id'] = isset($data['id']) ? $data['id'] : '';
                    $subscriptionData['status'] = 'success';

                    $subscriptionPayment = $this->subscriptionPayment->insertPayment($subscriptionData);
                    $this->logger->info("processSchedule subscriptionPayment result", $subscriptionPayment);
                } else {
                    $this->logger->info("payment/response createSchedule failed", $returnMessage);
                    return false;
                }
            }
            return true;
        } catch (\Exception $e) {
            $this->logger->info("payment/response processSchedule failed", $e->getMessage());
            return false;
        }
    }

    public function cek()
    {
        $quoteId = 4;
        $newQuote = $this->_quoteFactory->create();
        $newQuote = $newQuote->load($quoteId);
        $order = $this->_quoteManagement->submit($newQuote);

        // echo $order->getEntityId();
        $order->setState(\Magento\Sales\Model\Order::STATE_PROCESSING, true);
        $order->setStatus(\Magento\Sales\Model\Order::STATE_PROCESSING);
        $order->addStatusToHistory($order->getStatus(), 'Order processed successfully with reference');
        $order->save();

        echo $order->getEntityId();
    }

    public function createOrderSubscriptionFromWebhook($subscriptionData, $paymentStatus)
    {
        $this->logger->info('createOrderSubscriptionFromWebhook start creating order', $subscriptionData);
        $this->logger->info('createOrderSubscriptionFromWebhook paymentStatus', $paymentStatus);

        $data = $subscriptionData;
        $quoteId = $data['quote_id'];
        $amount = $paymentStatus['amount'];
        $paymentId = $paymentStatus['id'];

        $newQuote = $this->_quoteFactory->create();
        $newQuote = $newQuote->load($quoteId);
        $this->logger->info('createOrderSubscriptionFromWebhook newQuote', $newQuote);

        $order = $this->_quoteManagement->submit($newQuote);
        $this->logger->info('createOrderSubscriptionFromWebhook order', $order);

        $payment = $order->getPayment();

        $payment->setAmountPaid($amount)
            ->setLastTransId($paymentId)
            ->setTransactionId($paymentId)
            ->setIsTransactionClosed(true)
            ->setShouldCloseParentTransaction(true);

        $payment->setAdditionalInformation('TRANSACTION_ID', $paymentStatus['merchantTransactionId']);
        $payment->setAdditionalInformation('REFERENCE_ID', $paymentStatus['id']);
        $payment->setAdditionalInformation('CURRENCY', $paymentStatus['currency']);
        $payment->setAdditionalInformation('AMOUNT', $paymentStatus['amount']);
        $payment->setAdditionalInformation('PAYMENT_TYPE', $paymentStatus['paymentType']);
        $payment->setAdditionalInformation('RESULT_CODE', $paymentStatus['result']['code']);
        $payment->setAdditionalInformation('RESULT_DESCRIPTION', $paymentStatus['result']['description']);
        $payment->setAdditionalInformation('RISK_SCORE', $paymentStatus['risk']['score']);

        $payment->setAdditionalInformation('ORDER_STATUS_CODE', $paymentStatus['paymentType']);

        $orderStatus = $this->setOrderStatus($paymentStatus);
        if ($orderStatus) {
            $order->setState('new')->setStatus($orderStatus)->save();
            $order->addStatusToHistory($orderStatus, '', true)->save();
            $order->save();
        } else {
            $order->setState(\Magento\Sales\Model\Order::STATE_PROCESSING, true);
            $order->setStatus(\Magento\Sales\Model\Order::STATE_PROCESSING);
            $order->addStatusToHistory($order->getStatus(), 'Order processed successfully with reference');
            $order->save();

            // Add Invoice
            $invoiceService = $this->_objectManager->create('Magento\Sales\Model\Service\InvoiceService');

            $invoice = $invoiceService->prepareInvoice($order);
            $invoice->setRequestedCaptureCase(\Magento\Sales\Model\Order\Invoice::CAPTURE_ONLINE);
            $invoice->register();
            $invoice->getOrder()->setCustomerNoteNotify(false);
            $invoice->getOrder()->setIsInProcess(true);

            $transactionSave = $this->_objectManager->create('Magento\Framework\DB\Transaction');
            $transactionSave->addObject($invoice)->addObject($invoice->getOrder())->save();

            $invoiceSender = $this->_objectManager->create('Magento\Sales\Model\Order\Email\Sender\InvoiceSender');
            $invoiceSender->send($invoice);
        }

        //disable the quote
        $newQuote->setIsActive(0)->save();

        return $order;
    }
}
