<?php
/**
 * Viveum Plugin Installation Process
 *
 * The class is used for General Setting Tabs
 * Copyright (c) Viveum
 *
 * @class      Viveum_General_Settings
 * @package  Viveum/Classes
 * @located at  /includes/admin/
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * The class is used for General Setting Tabs
 */
class Viveum_General_Settings {

	/**
	 * Tab name
	 *
	 * @var string
	 */
	public static $tab_name      = 'viveum_settings';
	/**
	 * Mandatory field
	 *
	 * @var array
	 */
	public static $mandatory_field  = array(
		// Commented because only used for version tracker
		// 'viveum_general_merchant_email',
		// 'viveum_general_merchant_no',
		// 'viveum_general_shop_url',
		'viveum_general_merchant_location',
		'viveum_access_token',
	);

	/**
	 * Main function of the class
	 */
	public static function init() {
		$page_request = Viveum_General_Functions::viveum_get_request_value( 'page' );
		$tab_request = Viveum_General_Functions::viveum_get_request_value( 'tab' );
		if ( 'wc-settings' === $page_request && 'viveum_settings' === $tab_request ) { // safe request.
			self::viveum_save_tab_settings();
		}

		add_filter( 'woocommerce_settings_tabs_array',array( __CLASS__, 'add_settings_tab' ), 50 );
		add_action( 'woocommerce_settings_tabs_viveum_settings', array( __CLASS__, 'add_settings_page' ) );
		add_action( 'woocommerce_update_options_viveum_settings', array( __CLASS__, 'update_settings' ) );
	}

	/**
	 * Add Viveum tab to WooCommerce
	 * Calls from the hook "woocommerce_settings_tabs_array"
	 *
	 * @param array $woocommerce_tab WooCommerce tab.
	 * @return array $woocommerce_tab
	 */
	public static function add_settings_tab( $woocommerce_tab ) {
		$woocommerce_tab[ self::$tab_name ] = 'Viveum ' . __( 'General Setting', 'wc-viveum' );
		return $woocommerce_tab;
	}

	/**
	 * Add setting fields to the general setting page
	 * Calls from the hook "woocommerce_settings_tabs_{tab_name}"
	 */
	public static function add_settings_page() {
		woocommerce_admin_fields( self::viveum_settings_fields() );
	}

	/**
	 * Update setting fields to the general setting page
	 * Calls from the hook "woocommerce_update_options_{tab_name}"
	 */
	public static function update_settings() {
		woocommerce_update_options( self::viveum_settings_fields() );
	}

	private static function get_webhook_setting_desc() {
		if ( Viveum_General_Functions::is_php_version_lower_than_71() ) {
			if ( !Viveum_General_Functions::is_sodium_crypto_aead_accessible() ) {
				return "<div class='updated woocommerce-message' style='border-left-color: red !important'><p>Because your PHP version is lower than 7.1.*, <a href='https://www.php.net/manual/en/sodium.installation.php' target='_blank'>Libsodium</a> need to be activated for Webhook.</p></div>";
			}
		}

		return "";
	}

	/**
	 * Add setting fields for Viveum general setting
	 */
	public static function viveum_settings_fields() {
		global $viveum_payments;

		$settings = apply_filters( 'woocommerce_' . self::$tab_name, array(
			array(
				'title' => 'Viveum ' . __( 'General Setting', 'wc-viveum' ),
				'id'    => 'viveum_general_settings',
				'desc'  => '',
				'type'  => 'title',
			),
			array(
				'title' => __( 'Access Token', 'wc-viveum' ),
				'id'    => 'viveum_access_token',
				'type'  => 'text',
			),
			array(
				'title' => __( 'Recurring', 'wc-viveum' ),
				'id'    => 'viveum_general_recurring',
				'type'  => 'select',
				'options' => array(
					'0' => __( 'No', 'wc-viveum' ),
					'1' => __( 'Yes', 'wc-viveum' ),
				),
				'default' => '1',
			),
			// Commented because only used for version tracker
			// array(
			// 	'title' => __( 'Merchant Email', 'wc-viveum' ) . ' * ',
			// 	'id'    => 'viveum_general_merchant_email',
			// 	'css'   => 'width:25em;',
			// 	'type'  => 'text',
			// ),
			// array(
			// 	'title' => __( 'Merchant No. (Viveum)', 'wc-viveum' ) . ' * ',
			// 	'id'    => 'viveum_general_merchant_no',
			// 	'css'   => 'width:25em;',
			// 	'type'  => 'text',
			// 	'desc'  => __( 'Your Customer ID from Viveum', 'wc-viveum' ),
			// ),
			// array(
			// 	'title' => __( 'Shop URL', 'wc-viveum' ) . ' * ',
			// 	'id'    => 'viveum_general_shop_url',
			// 	'css'   => 'width:25em;',
			// 	'type'  => 'text',
			// ),

			// array(
			// 	'title' => __( 'Version Tracker', 'wc-viveum' ),
			// 	'id'    => 'viveum_general_version_tracker',
			// 	'css'   => 'width:25em; padding: 1px;',
			// 	'type'  => 'select',
			// 	'desc'  => __( 'When enabled, you accept to share your IP, email address, etc with Viveum.', 'wc-viveum' ),
			// 	'options' => array(
			// 		'0' => __( 'No', 'wc-viveum' ),
			// 		'1' => __( 'Yes', 'wc-viveum' ),
			// 	),
			// 	'default' => '1',
			// ),
			array(
				'title' => __( 'Enable Date of Birth and Gender Fields', 'wc-viveum' ),
				'id'    => 'viveum_general_dob_gender',
				'type'  => 'select',
				'desc'  => __( 'If enabled, date of birth and gender fields will be required in customer billing information. Those parameters are required by Klarna.', 'wc-viveum' ),
				'options' => array(
					'0' => __( 'No', 'wc-viveum' ),
					'1' => __( 'Yes', 'wc-viveum' ),
				),
				'default' => '0',
			),
			array(
				'title' => __( 'Merchant Location', 'wc-viveum' ) . ' * ',
				'id'    => 'viveum_general_merchant_location',
				'type'  => 'text',
				'desc'  => __( 'Principal place of business (Company Name, Address including the Country)', 'wc-viveum' ),
			),
			array(
				'type' => 'sectionend',
				'id' => 'viveum_general_settings',
			),

			array(
				'title' => __( 'Webhook Setting', 'wc-viveum' ),
				'id'    => 'viveum_webhook_setting',
				'desc'  => self::get_webhook_setting_desc(),
				'type'  => 'title',
			),
			array(
				'title' => __( 'Secret for url', 'wc-viveum' ),
				'id'    => 'viveum_webhook_url',
				'type'  => 'text',
				'value' => home_url() . '/wp-json/viveum-woocommerce/v1/webhook-receiver',
				'custom_attributes' => array(
					'readonly' => true,
				),
				'desc'  => __( 'Put this URL onto webhook setting on your BIP', 'wc-viveum' ),
			),
			array(
				'title' => __( 'Secret for encryption', 'wc-viveum' ),
				'id'    => 'viveum_secret_for_encryption',
				'type'  => 'text',
				'desc'  => __( 'Fill it with "Secret for encryption" that you have set on BIP', 'wc-viveum' ),
			),
			array(
				'type' => 'sectionend',
				'id' => 'viveum_vendor_script',
			),
		) );
		return apply_filters( 'woocommerce_' . self::$tab_name, $settings );
	}

	/**
	 * Redirect and show error message if empty mandatory fields when save the general settings
	 */
	public static function viveum_save_tab_settings() {
		$save_request = Viveum_General_Functions::viveum_get_request_value( 'save' );
		if ( $save_request ) {
			$is_fill_mandatory_fields = self::viveum_is_fill_mandatory_fields( $_REQUEST ); // input var okay.

			if ( ! $is_fill_mandatory_fields ) {
				$get = isset( $_GET ) ? $_GET : null; // input var okay.
				$redirect = get_admin_url() . 'admin.php?' . http_build_query( $get, '', '&' );
				$redirect = remove_query_arg( 'save' );
				$error_message = __( 'Please fill in all the mandatory fields', 'wc-viveum' );
				$redirect = add_query_arg( 'wc_error', rawurlencode( esc_attr( $error_message ) ), $redirect );
				wp_safe_redirect( $redirect );
				exit();
			} else {
				$redirect = get_admin_url() . 'admin.php?' . http_build_query( $_GET, '', '&' ); // input var okay.
				$redirect = remove_query_arg( 'wc_error' );
				$message = __( 'Your settings have been saved.', 'woocommerce' );
				$redirect = add_query_arg( 'wc_message', rawurlencode( esc_attr( $message ) ), $redirect );
				wp_safe_redirect( $redirect );
			}
		}
	}

	/**
	 * Validate mandatory fields from the general settings
	 *
	 * @param array $request from $_REQUEST.
	 * @return mixed
	 */
	public static function viveum_is_fill_mandatory_fields( $request ) {

		foreach ( $request as $fields_name => $field_value ) {
			if ( in_array( $fields_name, self::$mandatory_field, true ) ) {
				if ( trim( $field_value ) === '' ) {
					return false;
				}
			}
		}

		return true;
	}
}

Viveum_General_Settings::init();
