<?php
/**
 * Plugin Name: Viveum - WooCommerce
 * Plugin URI:  hhttps://www.viveum.com/
 * Description: WooCommerce with Viveum payment gateway
 * Author:      Viveum
 * Author URI:  https://www.viveum.com/
 * Version:     1.0.2
 *
 * @package     Viveum/Classes
 */

/**
 * Copyright (c) Viveum
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

require_once dirname( __FILE__ ) . '/viveum-install.php';
require_once dirname( __FILE__ ) . '/viveum-additional.php';
register_activation_hook( __FILE__, 'viveum_activation_process' );
register_deactivation_hook( __FILE__, 'viveum_deactivation_process' );
register_uninstall_hook( __FILE__, 'viveum_uninstallation_process' );
add_action( 'plugins_loaded', 'init_payment_gateway', 0 );

ob_start();

define( 'VIVEUM_VERSION', '1.0.2' );
define( 'VIVEUM_PLUGIN_FILE', __FILE__ );
define( 'VIVEUM_API_NAMESPACE', 'viveum-woocommerce/v1' );


/**
 * Check if WooCommerce is active, and if it isn't, disable Viveum payments.
 */
function get_notice_woocommerce_activation() {

	echo '<div id="notice" class="error"><p>';
	echo '<a href="http://www.woothemes.com/woocommerce/" style="text-decoration:none" target="_new">WooCommerce </a>' . esc_attr( __( 'must be active for the plugin', 'wc-viveum' ) ) . '<b> Viveum Payment Gateway for WooCommerce</b>';
	echo '</p></div>';
}

/**
 * Add configuration link at plugin installation
 *
 * @param array $links links.
 * @return array
 */
function add_configuration_links( $links ) {
	$add_links = array( '<a href="' . admin_url( 'admin.php?page=wc-settings&tab=viveum_settings' ) . '">' . __( 'General Setting', 'wc-viveum' ) . '</a>' );
	return array_merge( $add_links, $links );
}
add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), 'add_configuration_links' );

/**
 * Validate viveum term
 */
function validate_viveum_term() {
	if ( ! get_option( 'viveum_term' ) ) {
		echo '<div id="notice" class="updated"><p>';
		echo esc_attr( Viveum_General_Functions::viveum_translate_viveum_term( 'VIVEUMECOMMERCE_TT_VERSIONTRACKER' ) );
		echo ' <a href="' . esc_attr( admin_url( 'admin.php?page=wc-settings&tab=viveum_settings' ) ) . '">' . esc_attr( Viveum_General_Functions::viveum_translate_viveum_term( 'VIVEUMECOMMERCE_BACKEND_BT_ADMIN' ) ) . '</a>';
		echo '</p></div>';
		add_option( 'viveum_term', true );
	}
}

/**
 * Init payment gateway
 */
function init_payment_gateway() {

	/**
	 * Loads the Viveum language translation strings
	 */
	load_plugin_textdomain( 'wc-viveum', false, dirname( plugin_basename( __FILE__ ) ) . '/languages/' );

	if ( ! class_exists( 'WC_Payment_Gateway' ) ) {
		add_action( 'admin_notices', 'get_notice_woocommerce_activation' );
		return;
	} else {
		add_action( 'admin_notices', 'validate_viveum_term' );
	}

	include_once dirname( __FILE__ ) . '/includes/class-viveum-general-functions.php';
	include_once dirname( __FILE__ ) . '/includes/class-viveum-general-models.php';
	include_once dirname( __FILE__ ) . '/includes/admin/class-viveum-general-settings.php';
	include_once dirname( __FILE__ ) . '/includes/myaccount/class-viveum-payment-information.php';
	include_once dirname( __FILE__ ) . '/includes/core/class-viveumpaymentcore.php';
	include_once dirname( __FILE__ ) . '/includes/core/class-viveumversiontracker.php';
	include_once dirname( __FILE__ ) . '/includes/api/class-viveum-api.php';

	if ( ! class_exists( 'Viveum_Payment_Gateway' ) ) {

		/**
		 * Viveum Payment Gateway
		 *
		 * @class Viveum_Payment_Gateway
		 */
		class Viveum_Payment_Gateway extends WC_Payment_Gateway {

			/**
			 * Payment id
			 *
			 * @var string $payment_id
			 */
			protected $payment_id;

			/**
			 * Payment type
			 *
			 * @var string $payment_type
			 */
			protected $payment_type = 'DB';

			/**
			 * Payment brand
			 *
			 * @var string $payment_brand
			 */
			protected $payment_brand;

			/**
			 * Payment group
			 *
			 * @var string $payment_group
			 */
			protected $payment_group;
			/**
			 * Is recurring payment or not
			 *
			 * @var bool $is_recurring
			 */
			protected $is_recurring = false;

			/**
			 * When doing payment in TEST system. use this value as testMode parameter.
			 *
			 * @var string $test_mode
			 */
			protected $test_mode = 'INTERNAL';

			/**
			 * Payment display
			 *
			 * @var string $payment_display
			 */
			protected $payment_display = 'form';

			/**
			 * Languange
			 *
			 * @var string $language
			 */
			protected $language;

			/**
			 * Saved meta boxes
			 *
			 * @var bool $saved_meta_boxes
			 */
			private static $saved_meta_boxes = false;

			/**
			 * Added meta boxes
			 *
			 * @var bool $added_meta_boxes
			 */
			private static $added_meta_boxes = false;

			/**
			 * Added payment method
			 *
			 * @var bool $added_payment_method
			 */
			private static $added_payment_method = false;

			/**
			 * Updated meta boxes
			 *
			 * @var bool $updated_meta_boxes
			 */
			private static $updated_meta_boxes = false;

			/**
			 * Woocommerce order
			 *
			 * @var object $wc_order
			 */
			protected $wc_order;

			protected $without_redirect = false;

			/**
			 * Main function
			 */
			public function __construct() {
				$this->payment_id  = $this->id;
				$this->language    = Viveum_General_Functions::viveum_get_shop_language();
				$this->plugins_url = Viveum_General_Functions::viveum_get_plugin_url();

				$this->form_fields  = $this->get_backend_configuration_form_fields();
				$this->method_title = 'Viveum - ' . $this->get_title();
				$this->init_settings();

				$this->maybe_init_subscriptions();

				// Save admin configuration from woocomerce checkout tab.
				add_action( 'woocommerce_update_options_payment_gateways_' . $this->id, array( $this, 'process_admin_options' ) );
				// Frontend hook.
				add_action( 'before_woocommerce_pay', array( &$this, 'action_viveum_before_woocommerce_pay' ), 10, 0 );
				add_action( 'woocommerce_receipt_' . $this->payment_id, array( &$this, 'receipt_page' ) );
				add_action( 'woocommerce_thankyou_' . $this->payment_id, array( &$this, 'thankyou_page' ) );
				// Backend hook.
				add_action( 'woocommerce_process_shop_order_meta', array( &$this, 'save_order_meta' ) );
				add_action( 'woocommerce_admin_order_data_after_order_details', array( &$this, 'update_order_status' ) );
				add_action( 'woocommerce_admin_order_data_after_billing_address', array( &$this, 'add_payment_method' ) );
				add_action( 'woocommerce_admin_order_data_after_shipping_address', array( &$this, 'add_additional_information' ) );

				// Enable woocommerce refund for {payment gateway}.
				$this->supports = array( 'refunds' );

				if ( isset( WC()->session->viveum_thankyou_page ) ) {
					unset( WC()->session->viveum_thankyou_page );
				}
				if ( isset( WC()->session->viveum_receipt_page ) ) {
					unset( WC()->session->viveum_receipt_page );
				}
				if ( isset( WC()->session->viveum_confirmation_page ) ) {
					unset( WC()->session->viveum_confirmation_page );
				}
				if ( isset( WC()->session->viveum_display_error ) ) {
					$_POST['viveum_error'] = '1';
					unset( WC()->session->viveum_display_error );
				}
			}

			/**
			 * Get payment method backend configuration form fields
			 */
			public function get_backend_configuration_form_fields() {

				$form_fields = array(
					'enabled'     => array(
						'title'   => __( 'Enabled', 'wc-viveum' ),
						'type'    => 'checkbox',
						'default' => '',
					),
					'server_mode' => array(
						'title'   => __( 'Server', 'wc-viveum' ),
						'css'     => 'padding: 1px;',
						'type'    => 'select',
						'options' => array(
							'TEST' => __( 'TEST', 'wc-viveum' ),
							'LIVE' => __( 'LIVE', 'wc-viveum' ),
						),
						'default' => 'TEST',
					),
					'channel_id'  => array(
						'title'   => __( 'Entity-ID', 'wc-viveum' ),
						'type'    => 'text',
						'default' => '',
					),
				);

				return $form_fields;
			}

			/**
			 * Get Payment Type from backend configuration settings
			 * if not isset get from class attribute payment_type
			 *
			 * @return string
			 */
			public function get_payment_type() {

				if ( isset( $this->settings['trans_mode'] ) ) {
					return $this->settings['trans_mode'];
				}

				return $this->payment_type;
			}

			/**
			 * Get Payment Brand from backend configuration settings
			 * if not isset get from class attribute payment_brand
			 *
			 * @return string
			 */
			public function get_payment_brand() {
				if ( in_array( $this->payment_id, ['viveum_googlepay', 'viveum_applepay'] ) ) {
					return $this->payment_brand;
				}

				if ( isset( $this->settings['card_types'] ) && is_array( $this->settings['card_types'] ) ) {
					return strtoupper( implode( ' ', $this->settings['card_types'] ) );
				}

				return $this->payment_brand;
			}

			/**
			 * Get Payment Group from class attribute payment_group
			 *
			 * @return string
			 */
			public function get_payment_group() {

				return $this->payment_group;
			}

			/**
			 * Return true/false if payment method is recuring
			 *
			 * @return bool
			 */
			public function is_recurring() {
				return $this->is_recurring;
			}

			/**
			 * Get Credentials from Database
			 *
			 * @param bool $is_testmode_available is testmode available.
			 * @param bool $is_multichannel_available is testmode available.
			 * @return array
			 */
			public function get_credentials( $is_testmode_available = true, $is_multichannel_available = false ) {

				$credentials['access_token'] = get_option( 'viveum_access_token' );
				$credentials['server_mode']  = $this->settings['server_mode'];

				if ( ! empty( $this->settings['multichannel'] ) && $is_multichannel_available ) {
					$credentials['channel_id'] = $this->settings['channel_moto'];
				} else {
					$credentials['channel_id'] = $this->settings['channel_id'];
				}

				if ( $is_testmode_available ) {
					$credentials['test_mode'] = $this->get_test_mode();
				}

				return $credentials;
			}

			/**
			 * Get Test Mode from class attribue test_mode
			 * Return false if server mode setting is LIVE
			 *
			 * @return boolean | string ( EXTERNAL / INTERNAL )
			 */
			public function get_test_mode() {
				if ( 'LIVE' === $this->settings['server_mode'] ) {
					return false;
				}

				if ( in_array( $this->payment_id, ['viveum_googlepay', 'viveum_applepay', 'viveum_ccsaved', 'viveum_paypalsaved'] ) ) {
					return "EXTERNAL";
				}
				return $this->test_mode;
			}

			/**
			 * Get Payment Logo(s)
			 * ( extended at Viveum payment gateways class )
			 *
			 * @return boolean
			 */
			protected function viveum_get_payment_logo() {

				return false;
			}

			/**
			 * From class WC_Payment_Gateway
			 * Available payment gateway at frontend.
			 *
			 * @return boolean
			 */
			public function is_available() {

				$is_available = parent::is_available();

				if ( $is_available ) {
					$recurring = get_option( 'viveum_general_recurring' );
					switch ( $this->payment_id ) {
						case 'viveum_cc':
						case 'viveum_dd':
						case 'viveum_paypal':
							if ( $recurring && 0 < get_current_user_id() ) {
								$is_available = false;
							}
							break;
						case 'viveum_ccsaved':
						case 'viveum_ddsaved':
						case 'viveum_paypalsaved':
							if ( ! $recurring || 0 === get_current_user_id() ) {
								$is_available = false;
							}
							break;
						default:
							$is_available = true;
							break;
					}
				}
				return $is_available;
			}

			/**
			 * Check if current user login is admin
			 *
			 * @return boolean
			 */
			private function is_site_admin() {
				return in_array( 'administrator', wp_get_current_user()->roles, true );
			}

			/**
			 * Get wc order property value
			 *
			 * @param  string $property property of class WC_Order.
			 * @return mixed
			 */
			protected function get_wc_order_property_value( $property ) {
				if ( Viveum_General_Functions::is_version_greater_than( '3.0' ) ) {
					$function = 'get_' . $property;
					return $this->wc_order->$function();
				}
				return $this->wc_order->$property;
			}

			/**
			 * Get wc session property value
			 *
			 * @param string $property property of class WC_Session.
			 * @return mixed
			 */
			protected function get_wc_session_property_value( $property ) {
				$session_property_value = WC()->session->get( $property );

				if ( isset( $session_property_value ) ) {
					return $session_property_value;
				}
				return false;
			}

			/**
			 * From class WC_Payment_Gateway
			 * Payment gateway icon.
			 * ( extended at Viveum payment gateways class )
			 */
			public function get_icon() {

				$title = $this->get_title();

				$billing_fields_css_script = '<script>jQuery(".woocommerce-billing-fields").css({"min-height": "650px"});</script>';
				$icon_html                 = '<img src="' . $this->viveum_get_payment_logo() . '" alt="' . $title . '" title="' . $title . '" style="height:40px; margin:5px 10px 5px 0; float: none; vertical-align: middle; position: inherit;" />';
				if ( 'viveum_easycredit' === $this->payment_id ) {
					$total_amount = $this->get_order_total();
					$billing_dob  = $this->get_wc_session_property_value( 'viveum_billing_dob' );

					$icon_html .= '
					<script>
						jQuery(document).ready( function() {
							jQuery("#viveum_billing_dob").change(function() {
							});
						});
					</script>
					';

					$is_dob_valid            = Viveum_General_Functions::viveum_is_date_of_birth_valid( $billing_dob );
					$is_dob_lower_than_today = Viveum_General_Functions::viveum_is_date_of_birth_lower_than_today( $billing_dob );

					$is_amount_allowed = Viveum_General_Functions::viveum_is_amount_allowed( $total_amount );
					$gender            = $this->get_wc_session_property_value( 'viveum_billing_gender' );

					$error_message = '';
					if ( ! get_option( 'viveum_general_dob_gender' ) || ! $is_dob_valid ) {
						$error_message .= '<br />' . __( 'Please enter your date of birth in order to use easyCredit.', 'wc-viveum' );
					}
					if ( ! get_option( 'viveum_general_dob_gender' ) || ! $gender ) {
						$error_message .= '<br />' . __( 'Please enter your gender to make payment with easyCredit.', 'wc-viveum' );
					}
					if ( ! $is_amount_allowed ) {
						$error_message .= '<br />' . __( 'The financing amount is outside the permitted amounts (200 - 5,000 EUR)', 'wc-viveum' );
					}
					if ( ! $is_dob_lower_than_today ) {
						$error_message .= '<br />' . __( 'Date of birth should not be set in the future', 'wc-viveum' );
					}
					if ( ! Viveum_General_Functions::viveum_is_address_billing_equal_shipping() ) {
						$error_message .= '<br />' . __( 'In order to be able to pay with easyCredit, the delivery address must match the invoice address.', 'wc-viveum' );
					}
					if ( ! Viveum_General_Functions::viveum_is_shipping_method_chosen() ) {
						$error_message .= '<br />' . __( 'Please select a shipping method to use easyCredit', 'wc-viveum' );
					}
					$key = $this->payment_id;
					if ( $error_message ) {
						$icon_html .= $error_message;
						$icon_html .=
							'<script>
							document.getElementById("payment_method_' . $key . '").disabled = true;
							document.getElementById("payment_method_' . $key . '").checked = false;
							var payment_method_easycredit_class = document.getElementsByClassName("payment_method_' . $key . '")[0];
							if (payment_method_easycredit_class.style.opacity !== undefined) {
								payment_method_easycredit_class.style.opacity = "0.7";
							} else {
								payment_method_easycredit_class.style.filter = "alpha(opacity=70)";
							}
							</script>';
					} else {
						global $woocommerce;
						$customer = $woocommerce->customer;
						if ( version_compare( $woocommerce->version, '3.3', '>=' ) ) {
							$easycredit_button_script = 'jQuery("#place_order").html("' . __( 'Continue to Ratenkauf by easyCredit', 'wc-viveum' ) . '");';
						} else {
							$easycredit_button_script = 'jQuery("#place_order").val("' . __( 'Continue to Ratenkauf by easyCredit', 'wc-viveum' ) . '");';
						}
						$shop_name         = $this->settings['shop_name'];
						$billing_address_1 = $customer->get_billing_address_1();
						$billing_postcode  = $customer->get_billing_postcode();
						$billing_city      = $customer->get_billing_city();
						$frontend_term     = __( 'Yes, I would like to pay by installment and I agree that %shop_name% installment purchase of TeamBank AG (Partner of the Cooperative FiinanzGrupper Volksbanken Raiffeisenbanken), %billing_address_1%, %billing_postcode% %billing_city% gets for identity and credit checks and fraud prevention title and surname, date and place of birth, contact details ( Address, telephone, e-mail) as well as details of current and past orders and receives the test result for this purpose.', 'wc-viveum' );

						$icon_html .= '<div class="toc_easycredit" style="margin-left:3%;display:none;">
									    <dl style="margin-top: 0; margin-bottom: 20px;">
									        <dt style="float:left;">
									            <input id="toc_easycredit" type="checkbox" name="toc_easycredit" required="required">
									        </dt>
									        <dd>
									            <span>
									            ' . $this->sprintf_term(
							$frontend_term,
							array(
								'%shop_name%'         => $shop_name,
								'%billing_address_1%' => $billing_address_1,
								'%billing_postcode%'  => $billing_postcode,
								'%billing_city%'      => $billing_city,
							)
						) . '
									            </span>
									        </dd>
									    </dl>
									</div>';
						$icon_html .= '<script>
										jQuery(document).ready(function(){
			                            	var paymentMethods = document.querySelectorAll("[name=payment_method]");
			                            	var buttonCheckout = document.getElementById("place_order");
			                            	var tocEasyCredit = document.querySelector("#toc_easycredit");
			                            	document.querySelector("[name=payment_method]").click();
			                            	[].forEach.call(paymentMethods, function(elm) {
												elm.addEventListener("change", function() {
				                            		if (this.value == "viveum_easycredit") {
				                            			' . $easycredit_button_script . '
				                            			tocEasyCredit.style.display = "block";

				                            			tocEasyCredit.addEventListener("click", function() {
					                                       if (tocEasyCredit.checked == true) {
					                                            buttonCheckout.setAttribute("type", "submit");
					                                       } else {
					                                            buttonCheckout.setAttribute("type", "button");
					                                       }
					                                    });

					                                    if (tocEasyCredit.checked != true) {
					                                    	document.querySelector(".toc_easycredit").style.display = "block";
					                                    	buttonCheckout.setAttribute("type", "button");
					                                    	buttonCheckout.addEventListener("click", function() {
																if (tocEasyCredit.checked != true) {
																	document.getElementById("toc_easycredit").setCustomValidity("' . __( 'Please check this box if you want to proceed', 'wc-viveum' ) . '");
				                                        			document.querySelector("[name=checkout]").reportValidity();
																}
						                                    });
					                                    }
				                            		} else {
                                                                                tocEasyCredit.checked = false;
				                            			buttonCheckout.setAttribute("type", "submit");
				                            			document.querySelector(".toc_easycredit").style.display = "none";
				                            		}
				                            	});
											});
				                        });
										</script>';
					}// End if().
				}// End if().

				return apply_filters( 'woocommerce_gateway_icon', $billing_fields_css_script . $icon_html, $this->id );
			}

			/**
			 * Get translation from term of Condition of Easycredit
			 *
			 * @param string $str fill out the param with String.
			 * @param array  $vars fill out the param with Array.
			 * @return string
			 */
			protected function sprintf_term( $str, $vars ) {
				return str_replace( array_keys( $vars ), array_values( $vars ), $str );
			}

			/**
			 * Get multi icon ( brands ) such as Credit card
			 *
			 * @return string
			 */
			public function viveum_get_multi_icon() {
				$icon_html = '<br/>';

				$payment_setting = get_option( 'woocommerce_' . $this->payment_id . '_settings' );
				$cards           = $payment_setting['card_types'];
				if ( isset( $cards ) && '' !== $cards ) {
					foreach ( $cards as $card ) {
						$icon       = $this->plugins_url . '/assets/images/' . strtolower( $card ) . '.png';
						$icon_html .= '<img src="' . $icon . '" alt="' . strtolower( $card ) . '" title="' . strtolower( $card ) . '" style="height:40px; margin:5px 10px 5px 0; float: none; vertical-align: middle; display: inline-block;" />';
					}
				}

				return $icon_html;
			}

			/**
			 * Process the payment and return the result
			 *
			 * @param int $order_id order id.
			 * @return array
			 */
			public function process_payment( $order_id ) {
				$order = new WC_Order( $order_id );
				return array(
					'result'   => 'success',
					'redirect' => $order->get_checkout_payment_url( true ),
				);
			}

			/**
			 * Checking order status before create payment form
			 * Calls from hook "before_woocommerce_pay"
			 *
			 * @return void
			 */
			public function action_viveum_before_woocommerce_pay() {
				global $wp;

				$order_id = Viveum_General_Functions::viveum_get_query_string( 'order-pay', '' );
				$order    = new WC_Order( $order_id );

				if ( substr( $order->get_payment_method(), 0, 14 ) == 'viveum' ) {
					if ( $order->get_status() == 'processing'
						|| $order->get_status() == 'payment-accepted'
					) {
						sleep( 5 );
						wp_safe_redirect( $this->get_return_url( $order ) );
						exit();
					}
				}
			}

			/**
			 * Create payment gateways form ( form / redirect )
			 * Calls from hook "woocommerce_receipt_{gateway_id}"
			 *
			 * @param int $order_id order id.
			 */
			public function receipt_page( $order_id ) {
				$paypal_repeated   = Viveum_General_Functions::viveum_get_request_value( 'paypal_repeated' );
				$accept_payment    = Viveum_General_Functions::viveum_get_request_value( 'accept_payment' );
				$cancel_payment    = Viveum_General_Functions::viveum_get_request_value( 'cancel_payment' );
				$confirmation_page = Viveum_General_Functions::viveum_get_request_value( 'confirmation_page' );

				if ( Viveum_General_Functions::viveum_get_request_value( 'id' ) && empty( $paypal_repeated ) && empty( $accept_payment ) &&
					empty( $confirmation_page ) ) {
					$this->viveum_response_page( $order_id );
				} elseif ( ! empty( $paypal_repeated ) ) {
					$this->viveum_debit_paypal_recurring( $order_id );
				} elseif ( ! empty( $accept_payment ) ) {
					$this->viveum_accept_payment( $order_id );
				} elseif ( ! empty( $cancel_payment ) ) {
					$this->viveum_cancel_payment( $order_id );
				}

				if ( Viveum_General_Functions::viveum_get_request_value( 'confirmation_page' ) && ! isset( WC()->session->viveum_confirmation_page ) ) {
					$this->viveum_confirmation_page( $order_id );
					WC()->session->set( 'viveum_confirmation_page', true );
				} elseif ( ! Viveum_General_Functions::viveum_get_request_value( 'confirmation_page' ) && ! isset( WC()->session->viveum_receipt_page ) ) {
					$this->viveum_set_payment_form( $order_id );
					WC()->session->set( 'viveum_receipt_page', true );
				}
			}

			/**
			 * Set and load the payment checkout form
			 *
			 * @param int $order_id order id.
			 */
			private function viveum_set_payment_form( $order_id ) {
				global $wp;
				$is_one_click_payments = false;

				if ( get_option( 'viveum_general_dob_gender' ) ) {
					$billing_dob = $this->get_wc_session_property_value( 'viveum_billing_dob' );

					$is_dob_lower_than_today = Viveum_General_Functions::viveum_is_date_of_birth_lower_than_today( $billing_dob );
					$is_dob_valid            = Viveum_General_Functions::viveum_is_date_of_birth_valid( $billing_dob );
					if ( ! get_option( 'viveum_general_dob_gender' ) || ! $is_dob_valid ) {
						$this->viveum_do_error_payment( $order_id, 'wc-failed', 'ERROR_WRONG_DOB' );
					}
				}
				if ( 'viveum_easycredit' === $this->payment_id ) {
					$total_amount      = $this->get_order_total();
					$is_amount_allowed = Viveum_General_Functions::viveum_is_amount_allowed( $total_amount );
					$gender            = $this->get_wc_session_property_value( 'viveum_billing_gender' );
					if ( ! get_option( 'viveum_general_dob_gender' ) || ! $is_dob_valid ) {
						$this->viveum_do_error_payment( $order_id, 'wc-failed', 'ERROR_EASYCREDIT_PARAMETER_DOB' );
					} elseif ( ! get_option( 'viveum_general_dob_gender' ) || ! $gender ) {
						$this->viveum_do_error_payment( $order_id, 'wc-failed', 'ERROR_MESSAGE_EASYCREDIT_PARAMETER_GENDER' );
					} elseif ( ! $is_amount_allowed ) {
						$this->viveum_do_error_payment( $order_id, 'wc-failed', 'ERROR_MESSAGE_EASYCREDIT_AMOUNT_NOTALLOWED' );
					} elseif ( ! $is_dob_lower_than_today ) {
						$this->viveum_do_error_payment( $order_id, 'wc-failed', 'ERROR_EASYCREDIT_DOB_LOWER_THAN_TODAY' );
					} elseif ( ! Viveum_General_Functions::viveum_is_address_billing_equal_shipping() ) {
						$this->viveum_do_error_payment( $order_id, 'wc-failed', 'ERROR_EASYCREDIT_BILLING_NOTEQUAL_SHIPPING' );
					} elseif ( ! Viveum_General_Functions::viveum_is_shipping_method_chosen() ) {
						$this->viveum_do_error_payment( $order_id, 'wc-failed', 'ERROR_EASYCREDIT_NO_SHIPPING_METHOD' );
					}
				}

				Viveum_General_Functions::add_log( 'Get checkout parameters' );

				$checkout_id        = '';
				$payment_parameters = $this->viveum_set_payment_parameters( $order_id );

				if ( 'viveum_easycredit' === $this->payment_id ) {
					$checkout_result = ViveumPaymentCore::get_server_to_server_response( $payment_parameters );
					if ( $checkout_result['is_valid'] ) {
						$url_config['redirect_url'] = $checkout_result['response']['redirect']['url'];
						$payment_parameters         = $checkout_result['response']['redirect']['parameters'];
					}
				} else {
					$checkout_result = ViveumPaymentCore::get_checkout_result( $payment_parameters );
					if ( $checkout_result['is_valid'] ) {
						Viveum_General_Functions::add_log( 'Get payment widget url' );

						$url_config['payment_widget'] = ViveumPaymentCore::get_payment_widget_url( $payment_parameters['server_mode'], $checkout_result['response']['id'] );
					}
				}

				if ( ! $checkout_result['is_valid'] ) {
					$this->viveum_do_error_payment( $order_id, 'wc-canceled', $checkout_result['response'] );
				} elseif ( ! isset( $checkout_result['response']['id'] ) ) {
					$this->viveum_do_error_payment( $order_id, 'wc-canceled', 'ERROR_GENERAL_REDIRECT' );
				}

				if ( isset( $wp->request ) ) {
					$url_config['return_url'] = $this->viveum_get_home_url( $wp->request ) . 'key=' . Viveum_General_Functions::viveum_get_request_value( 'key' );
				} else {
					$url_config['return_url'] = get_page_link() . '&order-pay=' . Viveum_General_Functions::viveum_get_request_value( 'order-pay' ) . '&key=' . Viveum_General_Functions::viveum_get_request_value( 'key' );
				}

				$url_config['cancel_url'] = $this->get_wc_checkout_url();

				if ( ! empty( $payment_parameters['registrations'] ) ) {
					$is_one_click_payments = true;
				}

				if ( 'viveum_easycredit' === $this->payment_id ) {
					if ( isset( $payment_parameters[0]['value'] ) ) {
						$easyCreditUrl = $payment_parameters[0]['value'];
					} else {
						$easyCreditUrl = $url_config['redirect_url'];
					}
					$payment_widget_content = ViveumPaymentCore::get_payment_widget_content( $easyCreditUrl, $payment_parameters['server_mode'] );
				} else {
					$payment_widget_content = ViveumPaymentCore::get_payment_widget_content( $url_config['payment_widget'], $payment_parameters['server_mode'] );
				}

				if ( ! $payment_widget_content['is_valid'] || strpos( $payment_widget_content['response'], 'errorDetail' ) !== false ) {
					$this->viveum_do_error_payment( $order_id, 'wc-canceled', 'ERROR_GENERAL_REDIRECT' );
				}

				$payment_form = Viveum_General_Functions::viveum_get_payment_form( $this->payment_id );
				$args         = array(
					'url_config'            => $url_config,
					'payment_parameters'    => $payment_parameters,
					'plugins_url'           => $this->plugins_url,
					'is_one_click_payments' => $is_one_click_payments,
					'is_recurring'          => $this->is_recurring,
					'settings'              => $this->settings,
					'payment_method'        => $this->payment_id,
					'merchant_location'     => get_option( 'viveum_general_merchant_location' ),
					'payment_widget_script' => '<script type="text/javascript" src="' . $url_config['payment_widget'] . '"></script>',
				);

				if ( 'viveum_easycredit' !== $this->payment_id ) {
					wp_enqueue_script( 'viveum_formpayment_script', $url_config['payment_widget'], array(), null );
					wp_enqueue_style( 'viveum_formpayment_style', $this->plugins_url . '/assets/css/formpayment.css', array(), null );
				}

				switch ( $payment_form ) {
					case 'redirect':
						Viveum_General_Functions::viveum_include_template( dirname( __FILE__ ) . '/templates/checkout/template-redirect-payment.php', $args );
						break;
					case 'form_paypalsaved':
						if (  !$this->has_subscription( $order_id ) ) {
							$registered_paypal         = $this->viveum_get_registered_paypal();
							$args['registered_paypal'] = $registered_paypal;
						}
						Viveum_General_Functions::viveum_include_template( dirname( __FILE__ ) . '/templates/checkout/template-paypalsaved-form.php', $args );
						break;
					case 'servertoserver':
						Viveum_General_Functions::viveum_include_template( dirname( __FILE__ ) . '/templates/checkout/template-servertoserver-payment.php', $args );
						break;
					default:
						switch ( $this->payment_id ) {
							case 'viveum_googlepay':
								Viveum_General_Functions::viveum_include_template( dirname( __FILE__ ) . '/templates/checkout/template-google-pay-form.php', $args );
								break;
							case 'viveum_applepay':
								Viveum_General_Functions::viveum_include_template( dirname( __FILE__ ) . '/templates/checkout/template-apple-pay-form.php', $args );
								break;
							default:
								Viveum_General_Functions::viveum_include_template( dirname( __FILE__ ) . '/templates/checkout/template-payment-form.php', $args );
								break;
						}
						break;
				}
			}

			/**
			 * Get wc checkout url
			 *
			 * @return string
			 */
			protected function get_wc_checkout_url() {
				if ( Viveum_General_Functions::is_version_greater_than( '3.0' ) ) {
					return wc_get_checkout_url();
				} else {
					return WC()->cart->get_checkout_url();
				}
			}

			/**
			 * Return base url
			 *
			 * @param string $wp_request_type get url delimeter.
			 * @return string
			 */
			private function viveum_get_home_url( $wp_request_type ) {
				if ( false !== strpos( home_url( $wp_request_type ), '/?' ) ) {
					$home_url = home_url( $wp_request_type ) . '&';
				} else {
					$home_url = home_url( $wp_request_type ) . '/?';
				}
				return $home_url;
			}

			/**
			 * Get payment response page
			 *
			 * @param int $order_id order id.
			 */
			private function viveum_response_page( $order_id ) {
				$is_testmode_available = false;
				$status_parameters     = $this->get_credentials( $is_testmode_available );

				$id = Viveum_General_Functions::viveum_get_request_value( 'id' );

				Viveum_General_Functions::add_log( 'viveum_response_page() ID to look up: ', $id, $order_id );

				$payment_result = ViveumPaymentCore::get_payment_status( $id, $status_parameters );

				Viveum_General_Functions::add_log( 'Payment result ', $payment_result, $order_id );

				if ( ! $payment_result['is_valid'] ) {
					$this->viveum_do_error_payment( $order_id, 'wc-failed', $payment_result['response'], 'ERROR_GENERAL_NORESPONSE' );
				} else {
					if ( isset( $payment_result['response']['amount'] ) ) {
						$payment_amount = $payment_result['response']['amount'];
					}

					$result_status = ViveumPaymentCore::get_transaction_result( $payment_result['response']['result']['code'] );
					if ( 'PD' === $result_status && ! $this->is_payment_method_allowed_to_continue( $payment_result['response']['paymentBrand'] ) ) {
						$this->viveum_cancel_payment( $order_id );
					} elseif ( 'ACK' === $result_status || 'PD' === $result_status ) {
						if ( $this->is_recurring ) {
							$account = Viveum_General_Functions::viveum_get_account_by_result( $this->payment_id, $payment_result['response'] );
							if ( 'viveum_paypalsaved' === $this->payment_id ) {
								$registration_id                              = $payment_result['response']['id'];
								$payment_brand                                = $payment_result['response']['paymentBrand'];
								$payment_result                               = $this->viveum_debit_paypal_recurring( $order_id, $registration_id );
								$payment_result['response']['registrationId'] = $registration_id;
								$payment_result['response']['paymentBrand']   = $payment_brand;
							}

							Viveum_General_Functions::add_log( 'Payment is success' );

							$this->viveum_save_recurring_payment( $order_id, $payment_result['response'], $account );
						}
						$payment_result['response']['payment_status'] = $this->viveum_get_success_payment_status( $this->payment_id, $payment_result['response']['result']['code'], $payment_result['response']['paymentType'], $order_id );

						Viveum_General_Functions::add_log( 'Payment is success' );

						$this->viveum_do_success_payment( $order_id, $payment_result['response'] );
					} else {
						Viveum_General_Functions::add_log( 'Payment is failed' );

						$this->viveum_failed_response( $order_id, $payment_result['response'], $result_status );
					}
				}// End if().
			}

			/**
			 * Check if the payment method allowed to continue.
			 *
			 * @param String $payment_brand Payment Brand.
			 * @return boolean
			 */
			public function is_payment_method_allowed_to_continue( $payment_brand ) {
				switch ( $payment_brand ) {
					case 'SOFORTUEBERWEISUNG':
						return false;
					default:
						return true;
				}
			}

			/**
			 * Get payment response page
			 *
			 * @param int $order_id order id.
			 */
			private function viveum_confirmation_page( $order_id ) {
				$is_testmode_available = false;
				$status_parameters     = $this->get_credentials( $is_testmode_available );

				$id = Viveum_General_Functions::viveum_get_request_value( 'id' );

				$payment_result = ViveumPaymentCore::get_payment_server_to_server_status( $id, $status_parameters );

				if ( ! $payment_result['is_valid'] ) {
					$this->viveum_do_error_payment( $order_id, 'wc-failed', $payment_result['response'], 'ERROR_GENERAL_NORESPONSE' );
				} else {
					$result_status = ViveumPaymentCore::get_transaction_result( $payment_result['response']['result']['code'] );

					if ( 'ACK' === $result_status ) {
						$order = new WC_Order( $order_id );
						WC()->session->set(
							'easycredit_result',
							array(
								$id => $payment_result['response'],
							)
						);
						WC()->session->set( 'sum_of_interest', $payment_result['response']['resultDetails']['ratenplan.zinsen.anfallendeZinsen'] );
						WC()->session->set( 'order_total', $payment_result['response']['resultDetails']['ratenplan.gesamtsumme'] );
						WC()->session->set( 'tilgungsplan_text', $payment_result['response']['resultDetails']['tilgungsplanText'] );
						WC()->session->set( 'vorvertragliche_link', $payment_result['response']['resultDetails']['vorvertraglicheInformationen'] );

						add_action( 'woocommerce_thankyou_' . $this->payment_id, array( $this, 'get_woocommerce_thankyou_text' ) );
						add_filter( 'woocommerce_get_order_item_totals', array( $this, 'add_sumofinterest_ordertotal' ) );
						add_filter( 'woocommerce_thankyou_order_received_text', array( $this, 'get_woocommerce_thankyou_order_received_text' ) );
						add_action( 'woocommerce_thankyou', array( $this, 'get_woocommerce_thankyou' ) );
						add_filter( 'woocommerce_pay_order_button_text', array( $this, 'change_thankyou_to_confirmation' ) );

						wc_get_template(
							'checkout/thankyou.php',
							array(
								'order' => $order,
							)
						);
					} else {
						$this->viveum_failed_response( $order_id, $payment_result['response'], $result_status );
					}// End if().
				}// End if().
			}

			/**
			 * Get woocommerce thankyou text
			 */
			public function get_woocommerce_thankyou_text() {
				echo '<p>' . esc_attr( WC()->session->tilgungsplan_text ) . '</p>';
				echo "<p>
						<a href='" . esc_attr( WC()->session->vorvertragliche_link ) . "' target='_blank'>" . esc_attr( __( 'Vorvertragliche Informationen zum Ratenkauf hier abrufen', 'wc-viveum' ) ) . '</a>
					 </p>';
			}

			/**
			 * Get woocommerce thankyou order received text
			 */
			public function get_woocommerce_thankyou_order_received_text() {
				return __( 'Order Review', 'wc-viveum' );
			}

			/**
			 * Get woocommerce thankyou
			 */
			public function get_woocommerce_thankyou() {
				global $wp;
				if ( isset( $wp->request ) ) {
					$accept_payment_link = $this->viveum_get_home_url( $wp->request ) . 'key=' . Viveum_General_Functions::viveum_get_request_value( 'key' ) . '&accept_payment=1&id=' .
					Viveum_General_Functions::viveum_get_request_value( 'id' );
				} else {
					$accept_payment_link = get_page_link() . '&order-pay=' .
					Viveum_General_Functions::viveum_get_request_value( 'order-pay' ) . '&key=' .
					Viveum_General_Functions::viveum_get_request_value( 'key' ) . '&accept_payment=1&id=' .
					Viveum_General_Functions::viveum_get_request_value( 'id' );
				}

				$cancel_payment_link = $this->viveum_get_home_url( $wp->request ) . 'key=' . Viveum_General_Functions::viveum_get_request_value( 'key' ) . '&cancel_payment=1';

				// Input hidden is for make the input have the shame padding according to css style.
				echo '<div id="payment" style="padding-top: 10px; border-top: 1px solid #aaaaaa">
						<input type="hidden"/>
						<input type="submit" class="button alt"  id="place_order" onclick="window.location = \'' . esc_attr( $accept_payment_link ) . '\'" value="' . esc_attr( __( 'Place order', 'wc-viveum' ) ) . '" /></div>';
				echo '<script>
						var title = document.getElementsByClassName("woocommerce-thankyou-order-received")[0].innerHTML;
						if(typeof(document.getElementsByClassName("entry-title")[0]) != "undefined" && document.getElementsByClassName("entry-title")[0] != null) {
							document.getElementsByClassName("entry-title")[0].innerHTML = title;
						}
						var element = document.getElementsByClassName("order_details")[0];
						element.parentNode.removeChild(element);
					 </script>';
			}

			/**
			 * Add sum of interest and order total
			 *
			 * @param array $total_rows total rows from woocommerce.
			 * @return array
			 */
			public function add_sumofinterest_ordertotal( $total_rows ) {
				$total_rows['sum_of_interest']        = array(
					'label' => __( 'Sum of Interest', 'wc-viveum' ),
					'value' => wc_price(
						WC()->session->sum_of_interest,
						array(
							'currency' => get_woocommerce_currency(),
						)
					),
				);
				$total_rows['easycredit_order_total'] = array(
					'label' => __( 'Order Total', 'wc-viveum' ),
					'value' => wc_price(
						WC()->session->order_total,
						array(
							'currency' => get_woocommerce_currency(),
						)
					),
				);
				return $total_rows;
			}

			/**
			 * Accept payment after confirmation
			 *
			 * @param int $order_id order id.
			 */
			private function viveum_accept_payment( $order_id ) {
				$id = Viveum_General_Functions::viveum_get_request_value( 'id' );
				if ( ! isset( WC()->session->easycredit_result[ $id ] ) ) {
					$this->viveum_do_error_payment( $order_id, 'wc-failed', 'ERROR_GENERAL_CAPTURE_PAYMENT' );
				} else {
					$payment_result = WC()->session->easycredit_result[ $id ];
					unset( WC()->session->easycredit_result );
					$is_testmode_available     = true;
					$is_multichannel_available = true;

					if ( ViveumPaymentCore::set_number_format( $this->get_order_total() ) !== $payment_result['amount'] ) {
						$this->viveum_do_error_payment( $order_id, 'wc-failed', $capture_result['response'], 'ERROR_GENERAL_CAPTURE_PAYMENT' );
					}
					$capture_parameter                 = $this->get_credentials( $is_testmode_available, $is_multichannel_available );
					$capture_parameter['amount']       = $payment_result['amount'];
					$capture_parameter['currency']     = $payment_result['currency'];
					$capture_parameter['payment_type'] = 'CP';

					$capture_result = ViveumPaymentCore::back_office_operation( $id, $capture_parameter );

					if ( ! $capture_result['is_valid'] ) {
						$this->viveum_do_error_payment( $order_id, 'wc-failed', $capture_result['response'], 'ERROR_GENERAL_CAPTURE_PAYMENT' );
					} else {
						$capture_status = ViveumPaymentCore::get_transaction_result( $capture_result['response']['result']['code'] );
						if ( 'ACK' === $capture_status ) {
							$payment_result['payment_status'] = 'wc-payment-accepted';
							$this->viveum_do_success_payment( $order_id, $payment_result );
						} else {
							$this->viveum_failed_response( $order_id, $payment_result, $capture_status );
						}
					}
				}
			}

			/**
			 * Accept payment after confirmation
			 *
			 * @param int $order_id order id.
			 */
			public function viveum_cancel_payment( $order_id ) {
				$this->viveum_do_error_payment( $order_id, 'wc-cancelled', 'ERROR_GENERAL_CANCEL' );
			}

			/**
			 * Run after payment successful
			 *
			 * @param int   $order_id order id.
			 * @param array $payment_result payment result.
			 */
			public function viveum_do_success_payment( $order_id, $payment_result ) {
				
				$order         = new WC_Order( $order_id );
				$merchant_info = $this->viveum_get_merchant_info();

				// Not necessary at the moment for Viveum
				// if ( Viveum_General_Functions::viveum_is_version_tracker_active() ) {
				// 	ViveumVersionTracker::send_version_tracker( $merchant_info );
				// }

				$reference_id = $payment_result['id'];
				$this->viveum_save_transactions( $order_id, $payment_result, $reference_id );

				// Update Order Status.
				$order->update_status( 'wc-processing', 'order_note' );
				$order->update_status( $payment_result['payment_status'], 'order_note' );
				$order_awaiting_payment_session = WC()->session->order_awaiting_payment;
				// Empty awaiting payment session.
				if ( ! empty( $order_awaiting_payment_session ) ) {
					unset( WC()->session->order_awaiting_payment );
				}
				// Reduce stock levels.
				if ( Viveum_General_Functions::is_version_greater_than( '3.0' ) ) {
					wc_reduce_stock_levels( $order_id );
				} else {
					$order->reduce_order_stock();
				}

				if ( ! $this->without_redirect ) {
					// Remove cart.
					WC()->cart->empty_cart();

					wp_safe_redirect( $this->get_return_url( $order ) );
					exit();
				}
			}

			/**
			 * Run after payment response failed
			 *
			 * @param int    $order_id order id.
			 * @param array  $payment_result payment result.
			 * @param string $result_status result status.
			 */
			private function viveum_failed_response( $order_id, $payment_result, $result_status ) {
				if ( 'NOK' === $result_status ) {
					if ( 'viveum_easycredit' === $this->payment_id ) {
						$error_identifier = $payment_result;
					} else {
						$error_identifier = ViveumPaymentCore::get_error_identifier( $payment_result['result']['code'] );
					}
				} else {
					$error_identifier = 'ERROR_UNKNOWN';
				}

				$payment_result['payment_status'] = 'wc-failed';
				$this->viveum_save_transactions( $order_id, $payment_result, $payment_result['id'] );
				$this->viveum_do_error_payment( $order_id, $payment_result['payment_status'], $error_identifier );
			}

			/**
			 * Error payment action
			 *
			 * @param int          $order_id order id.
			 * @param string       $payment_status payment status.
			 * @param string|array $error_identifier error identifier.
			 * @param string       $error_default error default.
			 */
			public function viveum_do_error_payment( $order_id, $payment_status, $error_identifier, $error_default = 'ERROR_UNKNOWN' ) {
				global $woocommerce;

				$order = new WC_Order( $order_id );

				if ( 'ERROR_UNKNOWN' === $error_identifier ) {
					$error_identifier = $error_default;
				}

				if ( 'viveum_easycredit' === $this->payment_id && isset( $error_identifier['resultDetails']['Error'] ) ) {
					$easycredit_error_detail = $this->get_easycredit_error_detail( $error_identifier );
					if ( 'de' === $this->language ) {
						$error_translated = $easycredit_error_detail['renderedMessage'];
					} else {
						if ( isset( $error_identifier['resultDetails']['ExtendedDescription'] ) ) {
							$error_translated = $error_identifier['resultDetails']['ExtendedDescription'];
						} else {
							$error_translated = $easycredit_error_detail['renderedMessage'];
							if ( 'Die Adresse konnte nicht gefunden werden.' === $error_translated ) {
								$error_translated = 'The address could not be found.';
							}
						}
					}
					if ( isset( $easycredit_error_detail['field'] ) && 'null' !== $easycredit_error_detail['field'] ) {
						$error_translated = $easycredit_error_detail['field'] . ': ' . $error_translated;
					}
				} else {
					$error_translated = Viveum_General_Functions::viveum_translate_error_identifier( $error_identifier );
				}

				// Cancel the order.
				$order->update_status( $error_translated );
				$order->update_status( $payment_status, 'order_note' );


				if ( ! $this->without_redirect ) {
					// To display failure messages from woocommerce session.
					if ( isset( $error_translated ) ) {
						$woocommerce->session->errors = $error_translated;
						wc_add_notice( $error_translated, 'error' );
						WC()->session->set( 'viveum_display_error', true );
					}

					wp_safe_redirect( $this->get_wc_checkout_url() );
					exit();
				}
			}

			/**
			 * Get paypal recurring parameters
			 *
			 * @param int    $order_id order id.
			 * @param string $reg_id registration id.
			 * @return array
			 */
			private function viveum_get_debit_paypal_recurring_parameters( $order_id, $reg_id = false ) {
				$parameters                 = $this->viveum_set_payment_parameters( $order_id );
				$parameters['payment_type'] = $this->get_payment_type();
				unset( $parameters['payment_registration'] );

				if ( $reg_id ) {
					$parameters['payment_recurring'] = 'INITIAL';
				} else {
					$parameters['payment_recurring'] = 'REPEATED';
				}

				return $parameters;
			}

			/**
			 * Debit paypal recurring ( initial/repeated ) or use registered paypal account
			 *
			 * @param int    $order_id order id.
			 * @param string $reg_id registration id.
			 * @return : $paypal_result ( register paypal -> intial )
			 *            or
			 *            void ( repeated paypal )
			 */
			public function viveum_debit_paypal_recurring( $order_id, $reg_id = false ) {
				$payment_parameters = $this->viveum_get_debit_paypal_recurring_parameters( $order_id, $reg_id );

				if ( $reg_id ) {
					$registration_id = $reg_id;
				} else {
					$registration_id = Viveum_General_Functions::viveum_get_request_value( 'registrationId', '' );
				}

				$paypal_result = ViveumPaymentCore::use_registered_account( $registration_id, $payment_parameters );

				if ( ! $paypal_result['is_valid'] ) {
					$this->viveum_do_error_payment( $order_id, 'wc-failed', $paypal_result['response'], 'ERROR_GENERAL_NORESPONSE' );
				} else {
					$paypal_status = ViveumPaymentCore::get_transaction_result( $paypal_result['response']['result']['code'] );
					if ( 'ACK' === $paypal_status ) {
						if ( $reg_id ) {
							return $paypal_result;
						} else {
							$paypal_result['response']['payment_status'] = 'wc-payment-accepted';
							$this->viveum_do_success_payment( $order_id, $paypal_result['response'] );
						}
					} else {
						$this->viveum_failed_response( $order_id, $paypal_result['response'], $paypal_status );
					}
				}
			}

			/**
			 * Thankyou page
			 * Calls from hook "woocommerce_thankyou_{gateway_id}"
			 */
			public function thankyou_page() {
				if ( ! isset( WC()->session->viveum_thankyou_page ) ) {
					WC()->session->set( 'viveum_thankyou_page', true );
				}
			}

			/**
			 * [BACKEND] Save order from meta boxes
			 * Change payment status & save backend order
			 * Calls from hook "woocommerce_process_shop_order_meta"
			 */
			public function save_order_meta() {
				if ( ! self::$saved_meta_boxes ) {
					$original_post_status = Viveum_General_Functions::viveum_get_request_value( 'original_post_status', '' );
					$auto_draft           = Viveum_General_Functions::viveum_get_request_value( 'auto_draft', false );

					if ( 'auto-draft' === $original_post_status && '1' === $auto_draft ) {
						$this->viveum_save_backend_order();
					} else {
						$order_id                  = Viveum_General_Functions::viveum_get_request_value( 'post_ID', '' );
						$is_viveum_payment = $this->viveum_is_viveum_by_order( $order_id );
						if ( $is_viveum_payment ) {
							$this->viveum_change_payment_status();
						}
					}
					self::$saved_meta_boxes = true;
				}
			}

			/**
			 * [BACKEND] Change payment status at backend
			 */
			private function viveum_change_payment_status() {
				$order_id             = Viveum_General_Functions::viveum_get_request_value( 'post_ID', '' );
				$original_post_status = Viveum_General_Functions::viveum_get_request_value( 'original_post_status', '' );
				$order_post_status    = Viveum_General_Functions::viveum_get_request_value( 'order_status', '' );

				if ( 'wc-in-review' === $original_post_status && 'wc-in-review' === $order_post_status ) {
					$this->viveum_update_payment_status( $order_id, $original_post_status );
				} elseif ( 'wc-in-review' === $original_post_status && ( 'wc-failed' === $order_post_status || 'wc-cancelled' === $order_post_status ) ) {
					$this->viveum_update_payment_status( $order_id, $original_post_status, $order_post_status );
				} elseif ( 'wc-pre-authorization' === $original_post_status && 'wc-payment-accepted' === $order_post_status ) {
					$backoffice_config['payment_type']  = 'CP';
					$backoffice_config['order_status']  = $order_post_status;
					$backoffice_config['error_message'] = __( 'Unfortunately, your attempt to capture the payment failed.', 'wc-viveum' );

					$this->viveum_do_back_office_payment( $order_id, $backoffice_config );
				} elseif ( 'wc-payment-accepted' === $original_post_status && 'wc-refunded' === $order_post_status ) {
					$this->wc_order = new WC_Order( $order_id );
					$this->increase_order_stock();
					$backoffice_config['payment_type']  = 'RF';
					$backoffice_config['order_status']  = $order_post_status;
					$backoffice_config['error_message'] = __( 'Unfortunately, your attempt to refund the payment failed.', 'wc-viveum' );

					$this->viveum_do_back_office_payment( $order_id, $backoffice_config );
				} elseif ( 'wc-payment-accepted' !== $original_post_status && 'wc-refunded' === $order_post_status ) {
					$redirect = get_admin_url() . 'post.php?post=' . $order_id . '&action=edit';
					wp_safe_redirect( $redirect );
					exit;
				} elseif ( 'wc-failed' === $original_post_status ) {
					$redirect = get_admin_url() . 'post.php?post=' . $order_id . '&action=edit';
					WC_Admin_Meta_Boxes::add_error( __( 'Unfortunately, your transaction has failed. Please try again.', 'wc-viveum' ) );
					wp_safe_redirect( $redirect );
					exit;
				}

			}

			/**
			 * Check if order is using Viveum Payment module
			 * validate payment module from order id
			 *
			 * @param int $order_id order id.
			 * @return boolean
			 */
			private function viveum_is_viveum_by_order( $order_id ) {
				$transaction_log = Viveum_General_Models::viveum_get_db_transaction_log( $order_id );
				$payment_id      = $transaction_log['payment_id'];

				return $this->viveum_is_viveumecommerce_by_payment_id( $payment_id );
			}

			/**
			 * Check if payment method is using Viveum Payment
			 *
			 * @param string $payment_method payment method.
			 * @return boolean
			 */
			private function viveum_is_viveumecommerce_by_payment_id( $payment_method ) {
				if ( false !== strpos( $payment_method, 'viveum' ) ) {
					return true;
				}
				return false;
			}

			/**
			 * [BACKEND] Update payment status from gateway
			 *
			 * @param int    $order_id order id.
			 * @param string $original_post_status original post status.
			 * @param string $order_post_status order post status.
			 */
			private function viveum_update_payment_status( $order_id, $original_post_status = false, $order_post_status = false ) {
				$transaction_log = Viveum_General_Models::viveum_get_db_transaction_log( $order_id );

				if ( ! $original_post_status ) {
					$original_post_status = $transaction_log['payment_status'];
				}

				$payment_gateway  = wc_get_payment_gateway_by_order( $order_id );
				$transaction_data = $payment_gateway->get_credentials();

				$update_status_response = ViveumPaymentCore::update_status( $transaction_log['reference_id'], $transaction_data );

				if ( ! $update_status_response['is_valid'] ) {
					Viveum_General_Functions::add_log( 'update status is failed with error : ' . $update_status_response['response'] );
					return false;
				} else {
					$update_status_result_code = $update_status_response['response']['result']['code'];
					$transaction_result        = ViveumPaymentCore::get_transaction_result( $update_status_result_code );

					if ( 'ACK' === $transaction_result && ! ViveumPaymentCore::is_success_review( $update_status_result_code ) ) {
						$payment_type = $update_status_response['response']['paymentType'];
						if ( 'PA' === $payment_type ) {
							$payment_status = 'wc-pre-authorization';
						} elseif ( 'DB' === $payment_type ) {
							$payment_status = 'wc-payment-accepted';
						}

						Viveum_General_Models::viveum_update_db_transaction_log_status( $order_id, $payment_status );
						Viveum_General_Models::viveum_update_db_posts_status( $order_id, $payment_status );
						if ( $original_post_status !== $payment_status ) {
							if ( $order_post_status && ( 'wc-failed' === $order_post_status || 'wc-cancelled' === $order_post_status ) ) {
								$this->wc_order = new WC_Order( $order_id );
								$this->increase_order_stock();
							}
							$wc_order_status         = wc_get_order_statuses();
							$status_name['original'] = $wc_order_status[ $original_post_status ];
							$status_name['new']      = $wc_order_status[ $payment_status ];
							$this->viveum_add_order_notes( $order_id, $status_name );
						}
						Viveum_General_Functions::add_log( 'update status is success' );
						return true;
					} else {
						Viveum_General_Functions::add_log( 'update status is failed' );
						return false;
					}
				}// End if().
			}
			
			
			/**
			 * to initied subscription support and hooks.
			 *
			 * @return boolean
			 */
			public function maybe_init_subscriptions() {
				if ( ! $this->is_subscriptions_enabled() ) {
					return;
				}
		
				$this->supports = array_merge(
					$this->supports,
					[
						'subscriptions',
						'subscription_cancellation',
						'subscription_suspension',
						'subscription_reactivation',
						'subscription_amount_changes',
						'subscription_date_changes',
						'subscription_payment_method_change',
						'subscription_payment_method_change_customer',
						'subscription_payment_method_change_admin',
						'multiple_subscriptions',
					]
				);
		
				add_action( 'woocommerce_scheduled_subscription_payment_' . $this->id, [ $this, 'scheduled_subscription_payment' ], 10, 2 );
			}

			/**
			 * Checks if subscriptions are enabled on the site.
			 *
			 * @since 5.6.0
			 *
			 * @return bool Whether subscriptions is enabled or not.
			 */
			public function is_subscriptions_enabled() {
				return class_exists( 'WC_Subscriptions' ) && version_compare( WC_Subscriptions::$version, '2.2.0', '>=' );
			}

			/**
			 * Is $order_id a subscription?
			 *
			 * @since 5.6.0
			 *
			 * @param  int $order_id
			 * @return boolean
			 */
			public function has_subscription( $order_id ) {
				return ( function_exists( 'wcs_order_contains_subscription' ) && ( wcs_order_contains_subscription( $order_id ) || wcs_is_subscription( $order_id ) || wcs_order_contains_renewal( $order_id ) ) );
			}

			/**
			 * Scheduled_subscription_payment function.
			 *
			 * @param $amount_to_charge float The amount to charge.
			 * @param $renewal_order WC_Order A WC_Order object created to record the renewal payment.
			 */
			public function scheduled_subscription_payment( $amount_to_charge, $renewal_order ) {
				// Viveum_General_Functions::add_log( 'execute renewal order ', $renewal_order);
				Viveum_General_Functions::add_log( 'amount of renewal order ', $amount_to_charge);
				
				$this->process_subscription_payment( $amount_to_charge, $renewal_order, true, false );
			}

			/**
			 * Process_subscription_payment function.
			 *
			 * @since 3.0
			 * @since 4.0.4 Add third parameter flag to retry.
			 * @since 4.1.0 Add fourth parameter to log previous errors.
			 * @since 5.6.0 Process renewal payments for SEPA and UPE.
			 *
			 * @param float  $amount
			 * @param mixed  $renewal_order
			 * @param bool   $retry Should we retry the process?
			 * @param object $previous_error
			 */
			public function process_subscription_payment( $amount, $renewal_order, $retry = true, $previous_error = false ) {
				try {
					if ( wcs_order_contains_renewal( $renewal_order->id ) ) {
					
						Viveum_General_Functions::add_log( 'process subscription' );
						$renewal_transaction_log = Viveum_General_Models::viveum_get_db_transaction_log( $renewal_order->id );
						if ( !isset( $renewal_transaction_log['payment_status'] ) ) {
							if ( 'viveum_ccsaved' === $this->payment_id || 'viveum_paypalsaved' === $this->payment_id ) {
								Viveum_General_Functions::add_log( 'process subscription using viveum payment' );
								$parent_order_id = WC_Subscriptions_Renewal_Order::get_parent_order_id( $renewal_order->id );
								$parent_transaction_log = Viveum_General_Models::viveum_get_db_transaction_log( $parent_order_id );
								if ( !empty( $parent_transaction_log['additional_information'] ) ) {
									$this->viveum_do_renewal_payment( $amount, $renewal_order, $parent_order_id, $parent_transaction_log );
								}	else {
									throw new Exception("cant process because no register payment id on parent order");
								}
								
							} else {
								throw new Exception("cant process because not viveum payment");
							}
						}

				} else {
					Viveum_General_Functions::add_log( 'cant process non subscription order' );
				}

				} catch ( WC_Stripe_Exception $e ) {
					Viveum_General_Functions::add_log( 'Error: ' . $e->getMessage());
					$renewal_order->update_status( 'wc-failed', "failed renew order" );
					if ( is_callable( [ $renewal_order, 'save' ] ) ) {
						$renewal_order->save();
					}
				}
			}

			/**
			 * [BACKEND] Capture / Refund Payment
			 * do back office operations ( capture/refund )
			 *
			 * @param int   $order_id order id.
			 * @param array $back_office_config back office config.
			 * @param bool  $woocommerce_refund woocommerce refund.
			 * @return boolean ( if @param $woocommerce_refund = true )
			 */
			private function viveum_do_renewal_payment( $amount, $renewal_order, $parent_order_id, $transaction_log ) {

				$payment_gateway      = wc_get_payment_gateway_by_order( $renewal_order );
				$credentials = $payment_gateway->get_credentials();
				
				$order['order_id'] = $renewal_order->id;
				$order['channel_id'] = $credentials['channel_id'];
				$order['test_mode'] = 'EXTERNAL';
				$order['amount'] = $amount;
				$order['currency'] = $transaction_log['currency'];
				$order['payment_type'] = "DB";
				$order['merchantTransactionId'] = $renewal_order->id;
				
				$order['access_token'] = $credentials['access_token'];
				$subscription_additional_info = maybe_unserialize($transaction_log['additional_information']);
				
				if('viveum_paypalsaved' !== $this->payment_id) {	
					$order['standingInstruction.mode'] = "REPEATED";
					$order['standingInstruction.source'] = "MIT";
					$order['standingInstruction.type'] = "UNSCHEDULED";
					$order['standingInstruction.initialTransactionId'] = $subscription_additional_info['initialTransactionId'];

				}

				if('viveum_paypalsaved' === $this->payment_id) {
					$order['payment_recurring'] = "REPEATED";
				}



				
				$reg_id = $subscription_additional_info['registrationId'];
				
				

				Viveum_General_Functions::add_log( 'backend order parameters ', $order, $order['order_id'] );

				$order_result = ViveumPaymentCore::use_registered_account( $reg_id, $order );

				Viveum_General_Functions::add_log( 'backend order result ', $order_result );
				
				$transaction_result = ViveumPaymentCore::get_transaction_result( $order_result['response']['result']['code'] );

				if ( ! $order_result['is_valid'] ) {
					Viveum_General_Functions::add_log( 'renewal order is not success with error : ' . $order_result['response'] );
					throw new Exception("renewal order is not success");
				} else {
					if ( 'ACK' === $transaction_result ) {
						$order_result['response']['registrationId'] = $reg_id;
						$order_result['response']['payment_status'] = 'wc-payment-accepted';
						$this->without_redirect = true;
						$this->viveum_do_success_payment( $renewal_order->id, $order_result['response'] );
					} else {
						Viveum_General_Functions::add_log( 'renewal order is not ACK with error : ' . $order_result['response'] );
						throw new Exception("renewal order is not ACK with error");
					}
				}
			}

			/**
			 * [BACKEND] Save backend order
			 * Add order from backend with registered payment
			 */
			private function viveum_save_backend_order() {
				$order['order_id'] = Viveum_General_Functions::viveum_get_request_value( 'post_ID', '' );
				$reg_id            = Viveum_General_Functions::viveum_get_request_value( '_payment_recurring', '' );
				$payment_method    = Viveum_General_Functions::viveum_get_request_value( '_payment_method', '' );

				$is_viveum_payment = $this->viveum_is_viveumecommerce_by_payment_id( $payment_method );

				if ( $is_viveum_payment ) {
					$registered_payment     = Viveum_General_Models::viveum_get_db_registered_payment_by_regid( $reg_id );
					$order['payment_id']    = Viveum_General_Functions::viveum_get_payment_id_by_group( $registered_payment['payment_group'] );
					$order['user_id']       = $registered_payment['cust_id'];
					$order['payment_brand'] = $registered_payment['brand'];

					$order_parameter = $this->viveum_get_backend_order_parameters( $order );

					Viveum_General_Functions::add_log( 'backend order parameters ', $order_parameter, $order['order_id'] );

					$order_result = ViveumPaymentCore::use_registered_account( $reg_id, $order_parameter );

					Viveum_General_Functions::add_log( 'backend order result ', $order_result );

					if ( ! $order_result['is_valid'] ) {
						$order['payment_status'] = 'wc-failed';
						Viveum_General_Models::viveum_update_db_posts_status( $order['order_id'], $order['payment_status'] );

						Viveum_General_Functions::add_log( 'backend order is not success with error : ' . $order_result['response'] );
					} else {
						$transaction_result = ViveumPaymentCore::get_transaction_result( $order_result['response']['result']['code'] );

						if ( 'ACK' === $transaction_result ) {
							$order['payment_status'] = $this->viveum_get_success_payment_status( $order['payment_id'], $order_result['response']['result']['code'], $payment_result['response']['paymentType'], $order['order_id'] );
							$order['reference_id']   = $order_result['response']['id'];
							$this->viveum_do_success_backend_order( $order, $order_parameter );
							Viveum_General_Functions::add_log( 'backend order is success' );
						} else {
							$order['payment_status'] = 'wc-failed';
							Viveum_General_Models::viveum_update_db_posts_status( $order['order_id'], $order['payment_status'] );

							Viveum_General_Functions::add_log( 'backend order is not success' );
						}
					}

					$_POST['order_status']    = $order['payment_status'];
					$_POST['_payment_method'] = $order['payment_id'];
				}// End if().
			}

			/**
			 * [BACKEND] Get backend order parameters
			 * Get parameters for backend order
			 *
			 * @param array $order order.
			 * @return array
			 */
			private function viveum_get_backend_order_parameters( $order ) {
				$order_parameters = array();

				$payment_gateway  = Viveum_General_Functions::viveum_get_payment_gateway_by_id( $order['payment_id'] );
				$order_parameters = $payment_gateway->get_credentials();

				$order_detail = Viveum_General_Models::viveum_get_db_order_detail( $order['order_id'] );
				foreach ( $order_detail as $value ) {
					if ( '_order_total' === $value['meta_key'] ) {
						$order_parameters['amount'] = $value['meta_value'];
					} elseif ( '_order_currency' === $value['meta_key'] ) {
						$order_parameters['currency'] = $value['meta_value'];
					}
				}

				$order_parameters['transaction_id']    = $order['order_id'];
				$order_parameters['payment_type']      = $payment_gateway->get_payment_type();
				$order_parameters['payment_recurring'] = 'REPEATED';

				return $order_parameters;
			}

			/**
			 * [BACKEND] Run when backend order success
			 *
			 * @param array $order order.
			 * @param array $order_parameter order parameter.
			 */
			private function viveum_do_success_backend_order( $order, $order_parameter ) {
				$this->viveum_add_customer_note( $order['payment_id'], $order['order_id'] );
				$this->wc_order = new WC_Order( $order['order_id'] );

				$payment_gateway       = Viveum_General_Functions::viveum_get_payment_gateway_by_id( $order['payment_id'] );
				$order['payment_type'] = $payment_gateway->get_payment_type();
				$order['amount']       = $order_parameter['amount'];
				$order['currency']     = $order_parameter['currency'];
				if ( isset( $order_parameter['transaction_id'] ) ) {
					$order['transaction_id'] = $order_parameter['transaction_id'];
					$order['customer_id']    = ( $this->wc_order->get_user_id() ) ? $this->wc_order->get_user_id() : 0;
				}
				Viveum_General_Models::viveum_save_db_transaction( $order );
				Viveum_General_Models::viveum_update_db_posts_status( $order['order_id'], $order['payment_status'] );
			}

			/**
			 * [BACKEND] From class WC_Payment_Gateway
			 * Payment refund from button refund.
			 *
			 * @param int    $order_id order id.
			 * @param float  $amount amount.
			 * @param string $reason reason.
			 * @return boolean
			 */
			public function process_refund( $order_id, $amount = null, $reason = '' ) {
				$is_viveum_payment = $this->viveum_is_viveum_by_order( $order_id );

				if ( $is_viveum_payment ) {
					$woocommerce_refund                 = true;
					$backoffice_config['payment_type']  = 'RF';
					$backoffice_config['order_status']  = 'wc-refunded';
					$backoffice_config['error_message'] = __( 'Unfortunately, your attempt to refund the payment failed.', 'wc-viveum' );

					$refund_result = $this->viveum_do_back_office_payment( $order_id, $backoffice_config, $woocommerce_refund );

					if ( $refund_result ) {
						$order = new WC_Order( $order_id );
						$order->update_status( 'wc-refunded', 'order_note' );

						$args = array(
							'post_parent' => $order_id,
							'post_type'   => 'shop_order_refund',
						);

						$child_posts = get_children( $args );
						if ( Viveum_General_Functions::is_version_greater_than( '3.0' ) ) {
							$posts = $this->search_array_by_postmeta( $child_posts );
						} else {
							$posts = $this->search_array_by_value( 'Order Fully Refunded', $child_posts );
						}

						if ( !is_null( $posts ) && $posts ) {
							wp_delete_post( $posts->ID, true );
						}

						Viveum_General_Models::viveum_update_db_transaction_log_status( $order_id, 'wc-refunded' );
					}
					return $refund_result;
				}
				return false;
			}

			/**
			 * Search array by postmeta from multidimensional array
			 *
			 * @param  array $array array data.
			 * @return array | null
			 */
			public function search_array_by_postmeta( $array ) {
				foreach ( $array as $key => $val ) {
					if ( strpos( get_post_meta( $val->ID, '_refund_reason', true ), 'Bestellung vollständig' ) !== false
						|| strpos( get_post_meta( $val->ID, '_refund_reason', true ), 'Order fully' ) !== false ) {
						return $val;
					}
				}

				return null;
			}

			/**
			 * Search array by value from multidimensional array
			 *
			 * @param  string $value words you want to search for.
			 * @param  array  $array array data.
			 * @return array | null
			 */
			public function search_array_by_value( $value, $array ) {
				foreach ( $array as $key => $val ) {
					if ( $val->post_excerpt === $value ) {
						return $val;
					}
				}
				return null;
			}

			/**
			 * [BACKEND] Increase order stock
			 * increase order stock and add note to order detail
			 * used when refund
			 */
			protected function increase_order_stock() {
				$items = $this->wc_order->get_items();
				foreach ( $items as $item ) {
					$wc_product       = $this->wc_order->get_product_from_item( $item );
					$is_managed_stock = $this->is_managed_stock( $item['product_id'] );
					if ( $is_managed_stock ) {
						$product_stock = $wc_product->get_stock_quantity();
						$wc_product->increase_stock( $item['qty'] );
						$order_note = 'Item #' . $item['product_id'] .
							' stock increased from ' . $product_stock . ' to ' . ( $product_stock + 1 );
						$this->wc_order->add_order_note( $order_note );
					}
				}
			}

			/**
			 * [BACKEND] Check if the product stock is manageable
			 *
			 * @param int $product_id product id.
			 * @return bool
			 */
			protected function is_managed_stock( $product_id ) {
				$is_managed_stock = get_post_meta( $product_id, '_manage_stock', true );
				if ( 'yes' === $is_managed_stock ) {
					return true;
				}
				return false;
			}

			/**
			 * [BACKEND] Capture / Refund Payment
			 * do back office operations ( capture/refund )
			 *
			 * @param int   $order_id order id.
			 * @param array $back_office_config back office config.
			 * @param bool  $woocommerce_refund woocommerce refund.
			 * @return boolean ( if @param $woocommerce_refund = true )
			 */
			private function viveum_do_back_office_payment( $order_id, $back_office_config, $woocommerce_refund = false ) {
				$transaction_log           = Viveum_General_Models::viveum_get_db_transaction_log( $order_id );
				$is_testmode_available     = true;
				$is_multichannel_available = true;
				$amount                    = Viveum_General_Functions::viveum_get_request_value( 'refund_amount', false );

				$payment_gateway      = wc_get_payment_gateway_by_order( $order_id );
				$backoffice_parameter = $payment_gateway->get_credentials( $is_testmode_available, $is_multichannel_available );

				if ( $woocommerce_refund ) {
					$backoffice_parameter['amount'] = $amount;
				} else {
					$backoffice_parameter['amount'] = $transaction_log['amount'];
				}
				$backoffice_parameter['currency']     = $transaction_log['currency'];
				$backoffice_parameter['payment_type'] = $back_office_config['payment_type'];

				Viveum_General_Functions::add_log( 'Back office operation parameters ', $backoffice_parameter, $order_id );

				$backoffice_result = ViveumPaymentCore::back_office_operation( $transaction_log['reference_id'], $backoffice_parameter );

				Viveum_General_Functions::add_log( 'Back office operation result ', $backoffice_result, $order_id );

				if ( ! $backoffice_result['is_valid'] ) {
					if ( 'ERROR_UNKNOWN' !== $backoffice_result['response'] ) {
						$back_office_config['error_message'] = Viveum_General_Functions::viveum_translate_error_identifier( $backoffice_result['response'] );
					}
					if ( $woocommerce_refund ) {
						return false;
					} else {
						$this->viveum_backend_redirect_error( $order_id, $back_office_config['error_message'] );
					}
				} else {
					$transaction_result = ViveumPaymentCore::get_transaction_result( $backoffice_result['response']['result']['code'] );
					if ( 'ACK' === $transaction_result ) {
						Viveum_General_Models::viveum_update_db_transaction_log_status( $order_id, $back_office_config['order_status'] );

						Viveum_General_Functions::add_log( 'Back office operation is success' );
						if ( $woocommerce_refund ) {
							return true;
						}
					} else {
						if ( $woocommerce_refund ) {
							return false;
						} else {
							$this->viveum_backend_redirect_error( $order_id, $back_office_config['error_message'] );
						}
					}
				}
			}

			/**
			 * Backend redirect error
			 *
			 * @param int    $order_id order id.
			 * @param string $error_message error message.
			 */
			protected function viveum_backend_redirect_error( $order_id, $error_message ) {
				$redirect = get_admin_url() . 'post.php?post=' . $order_id . '&action=edit';
				WC_Admin_Meta_Boxes::add_error( $error_message );
				wp_safe_redirect( $redirect );
				exit;
			}

			/**
			 * [BACKEND] Add additional information
			 * add additional information on Edit Order Page
			 * Calls from hook "woocommerce_admin_order_data_after_shipping_address"
			 */
			public function add_additional_information() {
				if ( ! self::$added_meta_boxes ) {
					$additional_information = '';
					$order_id               = Viveum_General_Functions::viveum_get_request_value( 'post', false );

					$is_viveum_payment = $this->viveum_is_viveum_by_order( $order_id );
					if ( $is_viveum_payment ) {
						$transaction_log = Viveum_General_Models::viveum_get_db_transaction_log( $order_id );
						if ( isset( $transaction_log['additional_information'] ) ) {
							$additional_information = $this->viveum_set_additional_info( $transaction_log['additional_information'] );
						}
						$payment_gateway = wc_get_payment_gateway_by_order( $order_id );
						$args            = array(
							'payment_method_title'   => $payment_gateway->get_title(),
							'transaction_log'        => $transaction_log,
							'additional_information' => $additional_information,
						);

						if ( $this->has_subscription( $order_id ) ) {
							$args['isSubscription'] = true;
							$subscription_additional_info = maybe_unserialize($transaction_log['additional_information']);
							$args['additional_information'] = $subscription_additional_info['registrationId'];
						}
						// var_dump($args['transaction_log']); die();
						Viveum_General_Functions::viveum_include_template( dirname( __FILE__ ) . '/templates/admin/meta-boxes/template-additional-information.php', $args );
					}
					self::$added_meta_boxes = true;
				}
			}

			/**
			 * [BACKEND] Update order status
			 * update order status from gateway
			 * Calls from hook "woocommerce_admin_order_data_after_order_details"
			 */
			public function update_order_status() {
				$post_type = false;
				if ( isset( $_GET['post_type'] ) ) { // input var okay.
					$post_type = wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['post_type'] ) ) ); // input var okay.
				}

				if ( ! self::$updated_meta_boxes && 'shop_order' !== $post_type ) {
					$order_id                  = Viveum_General_Functions::viveum_get_request_value( 'post', false );
					$is_viveum_payment = $this->viveum_is_viveum_by_order( $order_id );

					if ( $is_viveum_payment ) {
						$order = wc_get_order( $order_id );
						if ( 'in-review' === $order->get_status() ) {
							$request_section = Viveum_General_Functions::viveum_get_request_value( 'section', false );

							if ( $order_id && 'update-order' === $request_section ) {
								$this->viveum_update_payment_status( $order_id, 'wc-in-review' );
								$redirect = get_admin_url() . 'post.php?post=' . $order_id . '&action=edit';
								wp_safe_redirect( $redirect );
								exit;
							}
							$args = array(
								'order_id' => $order_id,
							);
							Viveum_General_Functions::viveum_include_template( dirname( __FILE__ ) . '/templates/admin/meta-boxes/template-update-order.php', $args );
						}
					}
					self::$updated_meta_boxes = true;
				}
			}

			/**
			 * [BACKEND] Add payment method
			 * Add Payment Method ( Recurring ) at Backend Order
			 * Calls from hook "woocommerce_admin_order_data_after_billing_address"
			 */
			public function add_payment_method() {
				if ( ! self::$added_payment_method ) {
					$action   = Viveum_General_Functions::viveum_get_request_value( 'action', 'false' );
					$order_id = Viveum_General_Models::viveum_get_db_last_order_id();
					if ( $order_id ) {
						$args = array(
							'action' => $action,
						);
						Viveum_General_Functions::viveum_include_template( dirname( __FILE__ ) . '/templates/admin/meta-boxes/template-add-payment-method.php', $args );
					}

					self::$added_payment_method = true;
				}
			}

			/**
			 * Get success payment status
			 *
			 * @param string $payment_id payment id.
			 * @param string $payment_result_code payment result code.
			 * @return string
			 */
			private function viveum_get_success_payment_status( $payment_id, $payment_result_code, $payment_type_from_gateway, $order_id ) {
				if ( $this->has_subscription( $order_id ) ) {
					$payment_type   = $payment_type_from_gateway;
				} else {
					$payment_type   = Viveum_General_Functions::viveum_get_payment_gateway_by_id( $payment_id )->get_payment_type();
				}
				$is_success_review = ViveumPaymentCore::is_success_review( $payment_result_code );
				$result_status = ViveumPaymentCore::get_transaction_result( $payment_result_code );

				if ( $is_success_review ) {
					$payment_status = 'wc-in-review';
				} elseif ( $result_status === "PD" ) {
					$payment_status = 'wc-pending';
				} else {
					if ( 'PA' === $payment_type ) {
						$payment_status = 'wc-pre-authorization';
					} else {
						$payment_status = 'wc-payment-accepted';
					}
				}

				return $payment_status;
			}

			/**
			 * Add customer note
			 *
			 * @param string $payment_id payment id.
			 * @param int    $order_id order id.
			 */
			private function viveum_add_customer_note( $payment_id, $order_id ) {
				$this->wc_order = new WC_Order( $order_id );

				$customer_note = $this->get_wc_order_property_value( 'customer_note' );

				$new_line             = "\n";
				$transaction_id_title = __( 'Transaction ID', 'wc-viveum' );

				$payment_comments = $this->get_title() . $new_line;

				if ( $customer_note ) {
					$customer_note .= $new_line;
				}

				$customer_note .= html_entity_decode( $payment_comments, ENT_QUOTES, 'UTF-8' );
				$order_notes    = array(
					'ID'           => $this->get_wc_order_property_value( 'id' ),
					'post_excerpt' => $customer_note,
				);
				wp_update_post( $order_notes );
			}

			/**
			 * Get cart items parameters
			 *
			 * @return array
			 */
			private function viveum_set_cart_items_parameters() {
				global $woocommerce;
				$wc_cart    = $woocommerce->cart;
				$count      = 0;
				$cart_items = array();
				foreach ( $wc_cart->get_cart() as $cart ) {
					$cart_items[ $count ]['merchant_item_id'] = $cart['product_id'];
					if ( 'viveum_easycredit' !== $this->payment_id ) {
						$cart_items[ $count ]['discount'] = Viveum_General_Functions::viveum_get_payment_discount_in_percent( $cart );
					}
					$cart_items[ $count ]['quantity'] = (int) $cart['quantity'];
					if ( Viveum_General_Functions::is_version_greater_than( '3.0' ) ) {
						$cart_items[ $count ]['name'] = $cart['data']->get_name();
					} else {
						$cart_items[ $count ]['name'] = $cart['data']->get_title();
					}
					$cart_items[ $count ]['price'] = Viveum_General_Functions::viveum_get_payment_price_with_tax_and_discount( $cart );
					$cart_items[ $count ]['tax']   = Viveum_General_Functions::viveum_get_payment_tax_in_percent( $cart );

					$count++;
				}

				return $cart_items;
			}

			/**
			 * Set customer parameters for order
			 *
			 * @param int $order_id order id.
			 * @return array
			 */
			private function viveum_set_customer_parameters( $order_id ) {
				$this->wc_order = new WC_Order( $order_id );

				$customer['first_name'] = $this->get_wc_order_property_value( 'billing_first_name' );
				$customer['last_name']  = $this->get_wc_order_property_value( 'billing_last_name' );
				$customer['email']      = $this->get_wc_order_property_value( 'billing_email' );
				$customer['phone']      = $this->get_wc_order_property_value( 'billing_phone' );
				$customer['birthdate']  = get_option( 'viveum_general_dob_gender' ) ? date( 'Y-m-d', strtotime( $this->get_wc_session_property_value( 'viveum_billing_dob' ) ) ) : '';
				$customer['sex']        = get_option( 'viveum_general_dob_gender' ) ? Viveum_General_Functions::viveum_get_initial_gender( $this->get_wc_session_property_value( 'viveum_billing_gender' ) ) : '';

				return $customer;
			}

			/**
			 * Set billing parameters for order
			 *
			 * @param int $order_id order id.
			 * @return array
			 */
			private function viveum_set_billing_parameters( $order_id ) {
				$this->wc_order = new WC_Order( $order_id );

				$billing['street'] = $this->get_wc_order_property_value( 'billing_address_1' );
				if ( trim( $this->get_wc_order_property_value( 'billing_address_2' ) ) ) {
					$billing['street'] .= ', ' . $this->get_wc_order_property_value( 'billing_address_2' );
				}
				$billing['city']         = $this->get_wc_order_property_value( 'billing_city' );
				$billing['zip']          = $this->get_wc_order_property_value( 'billing_postcode' );
				$billing['country_code'] = $this->get_wc_order_property_value( 'billing_country' );

				return $billing;
			}

			/**
			 * Set shipping parameters for order
			 *
			 * @param int $order_id order id.
			 * @return array
			 */
			private function viveum_set_shipping_parameters( $order_id ) {
				$this->wc_order = new WC_Order( $order_id );

				$shipping['street'] = $this->get_wc_order_property_value( 'shipping_address_1' );
				if ( trim( $this->get_wc_order_property_value( 'shipping_address_2' ) ) ) {
					$shipping['street'] .= ', ' . $this->get_wc_order_property_value( 'shipping_address_2' );
				}
				$shipping['city']         = $this->get_wc_order_property_value( 'shipping_city' );
				$shipping['zip']          = $this->get_wc_order_property_value( 'shipping_postcode' );
				$shipping['country_code'] = $this->get_wc_order_property_value( 'shipping_country' );

				return $shipping;
			}

			/**
			 * Get PayDirekt parameter for order
			 *
			 * @return array
			 */
			private function viveum_get_paydirekt_parameters() {
				$paydirekt['PAYDIREKT_minimumAge']        = $this->settings['minimum_age'];
				$paydirekt['PAYDIREKT_payment.isPartial'] = $this->settings['payment_is_partial'];

				return $paydirekt;
			}

			/**
			 * Get Klarna parameter for order
			 *
			 * @return array
			 */
			private function viveum_get_klarna_parameters() {
				if ( 'viveum_klarnasliceit' === $this->payment_id ) {
					$klarna['KLARNA_PCLASS_FLAG'] = $this->settings['pclass'];
					if ( gettype( $klarna['KLARNA_PCLASS_FLAG'] ) == 'string' ) {
						$klarna['KLARNA_PCLASS_FLAG'] = trim( $klarna['KLARNA_PCLASS_FLAG'] );
					}
				}

				$klarna['KLARNA_CART_ITEM1_FLAGS'] = '32';

				return $klarna;
			}

			/**
			 * Get Klarna parameter for order
			 *
			 * @return array
			 */
			private function viveum_get_easycredit_parameters() {
				$easycredit['RISK_ANZAHLBESTELLUNGEN'] = Viveum_General_Functions::viveum_get_order_count();

				$easycredit['RISK_BESTELLUNGERFOLGTUEBERLOGIN'] = is_user_logged_in() ? 'true' : 'false';

				$easycredit['RISK_KUNDENSTATUS'] = Viveum_General_Functions::viveum_get_risk_kunden_status();

				$easycredit['RISK_KUNDESEIT'] = Viveum_General_Functions::viveum_get_customer_created_date();

				return $easycredit;
			}

			/**
			 * Set recurring parameters for order
			 *
			 * @param int $order_id order id.
			 * @return array
			 */
			private function viveum_set_recurring_parameters($order_id) {
				$recurring = array();
				if ( 'viveum_paypalsaved' !== $this->payment_id && !$this->has_subscription( $order_id ) ) {
					$recurring['registrations'] = $this->viveum_set_registration_parameters();
				}

				$recurring['payment_registration'] = 'true';
				if ( 'viveum_ccsaved' === $this->payment_id ) {
					$recurring['3D']['amount']   = $this->get_order_total();
					$recurring['3D']['currency'] = get_woocommerce_currency();
				}

				if ( $this->has_subscription( $order_id ) &&  'viveum_paypalsaved' !== $this->payment_id ) {
					$recurring['standingInstruction.mode']   = "INITIAL";
					$recurring['standingInstruction.source'] = "CIT";
					$recurring['standingInstruction.type'] = "UNSCHEDULED";
				}

				return $recurring;
			}

			/**
			 * Set payment parameters for order
			 *
			 * @param int $order_id order id.
			 * @return array
			 */
			private function viveum_set_payment_parameters( $order_id ) {
				$this->wc_order = new WC_Order( $order_id );
				
				$payment_parameters                   = $this->get_credentials();
				$payment_parameters['amount']         = $this->get_order_total();
				$payment_parameters['currency']       = get_woocommerce_currency();
				$payment_parameters['customer']       = $this->viveum_set_customer_parameters( $order_id );
				$payment_parameters['billing']        = $this->viveum_set_billing_parameters( $order_id );
				$payment_parameters['payment_type']   = $this->has_subscription( $order_id )?'DB':$this->get_payment_type();
				$payment_parameters['payment_brand']  = $this->get_payment_brand();
				$payment_parameters['merchantTransactionId']  = $order_id;
				$payment_parameters['customParameters'] = array(
					'wc_order_id' => $order_id,
					'viveum_payment_id' => $this->payment_id,
				);

				if ( $this->is_recurring ) {
					$recurring_parameter = $this->viveum_set_recurring_parameters($order_id);
					$payment_parameters  = array_merge( $payment_parameters, $recurring_parameter );
					if ( 'viveum_paypalsaved' === $this->payment_id ) {
						unset( $payment_parameters['payment_type'] );
					}
				}

				if ( 'viveum_paydirekt' === $this->payment_id ) {
					$payment_parameters['customParameters'] = array_merge(
						$payment_parameters['customParameters'],
						$this->viveum_get_paydirekt_parameters()
					);
				}

				if ( 'viveum_klarnapaylater' === $this->payment_id || 'viveum_klarnasliceit' === $this->payment_id ) {
					$payment_parameters['cartItems']        = $this->viveum_set_cart_items_parameters();
					$payment_parameters['customParameters'] = array_merge(
						$payment_parameters['customParameters'],
						$this->viveum_get_klarna_parameters()
					);
				}

				if ( 'viveum_easycredit' === $this->payment_id ) {
					$payment_parameters['cartItems']        = $this->viveum_set_cart_items_parameters();
					$payment_parameters['customParameters'] = array_merge(
						$payment_parameters['customParameters'],
						$this->viveum_get_easycredit_parameters()
					);
					$payment_parameters['shipping']         = $this->viveum_set_shipping_parameters( $order_id );

					global $wp;
					if ( isset( $wp->request ) ) {
						$payment_parameters['shopperResultUrl'] = $this->viveum_get_home_url( $wp->request ) . 'key=' . Viveum_General_Functions::viveum_get_request_value( 'key' ) . '&confirmation_page=1';
					} else {
						$payment_parameters['shopperResultUrl'] = get_page_link() . '&order-pay=' . Viveum_General_Functions::viveum_get_request_value( 'order-pay' ) . '&key=' . Viveum_General_Functions::viveum_get_request_value( 'key' ) . '&confirmation_page=1';
					}
				}

				Viveum_General_Functions::add_log( 'Checkout parameters ', $payment_parameters, $order_id );

				$payment_parameters['customer_ip'] = Viveum_General_Functions::get_customer_ip();
				
				return $payment_parameters;
			}

			/**
			 * Set serialize additional information

			 * @return serialize;
			 */
			private function viveum_set_serialize_add_info() {
				$additional_info = array();
				if ( isset( WC()->session->sum_of_interest ) ) {
					$additional_info[] = 'FRONTEND_EASYCREDIT_INTEREST=>' . WC()->session->sum_of_interest;
				}
				if ( isset( WC()->session->order_total ) ) {
					$additional_info[] = 'FRONTEND_EASYCREDIT_TOTAL=>' . WC()->session->order_total;
				}
				if ( 'viveum_easycredit' === $this->payment_id ) {
					unset( WC()->session->tilgungsplan_text, WC()->session->sum_of_interest, WC()->session->order_total );
				}
				$additional_info_serialize = maybe_serialize( $additional_info );

				unset(
					WC()->session->tilgungsplan_text,
					WC()->session->sum_of_interest,
					WC()->session->order_total
				);

				return $additional_info_serialize;
			}

			/**
			 * Set additional information
			 *
			 * @param serialize $serialize_info serialize info.
			 * @return array
			 */
			private function viveum_set_additional_info( $serialize_info ) {
				$additional_info = false;
				if ( $serialize_info ) {
					$unserialize_info = maybe_unserialize( $serialize_info );
					foreach ( $unserialize_info as $info ) {
						$explode_info                        = explode( '=>', $info );
						$additional_info[ $explode_info[0] ] = $explode_info[1];
					}
				}

				return $additional_info;
			}

			/**
			 * Set registered payment accounts for one click checkout order
			 *
			 * @return array
			 */
			private function viveum_set_registration_parameters() {

				$registration        = array();
				$credentials         = $this->get_credentials();
				$registered_payments = Viveum_General_Models::viveum_get_db_registered_payment( $this->payment_group, $credentials );

				foreach ( $registered_payments as $key => $registrations ) {
					$registration[ $key ] = $registrations['reg_id'];
				}

				return $registration;
			}

			/**
			 * Get registered paypal account for one click checkout order
			 *
			 * @return array
			 */
			private function viveum_get_registered_paypal() {

				$registered_paypal   = array();
				$credentials         = $this->get_credentials();
				$registered_payments = Viveum_General_Models::viveum_get_db_registered_payment( $this->payment_group, $credentials );

				foreach ( $registered_payments as $key => $registered_payment ) {
					$registered_paypal[ $key ]['reg_id']          = $registered_payment['reg_id'];
					$registered_paypal[ $key ]['email']           = $registered_payment['email'];
					$registered_paypal[ $key ]['payment_default'] = $registered_payment['payment_default'];
				}

				return $registered_paypal;
			}

			/**
			 * Save recurring payment from pay and save order
			 *
			 * @param int   $order_id order id.
			 * @param array $payment_result payment result.
			 * @param array $account account.
			 */
			public function viveum_save_recurring_payment( $order_id, $payment_result, $account ) {
				$this->wc_order = new WC_Order( $order_id );
				$is_registered_payment = Viveum_General_Models::viveum_is_registered_payment_db( $payment_result['registrationId'] );

				if ( ! $is_registered_payment ) {
					$credentials                           = $this->get_credentials();
					$registered_payment                    = $account;
					$registered_payment['cust_id']         = $this->wc_order->get_user_id();
					$registered_payment['payment_group']   = $this->payment_group;
					$registered_payment['payment_brand']   = $payment_result['paymentBrand'];
					$registered_payment['server_mode']     = $credentials['server_mode'];
					$registered_payment['channel_id']      = $credentials['channel_id'];
					$registered_payment['registration_id'] = $payment_result['registrationId'];
					$registered_payment['payment_default'] = Viveum_General_Models::viveum_get_db_payment_default( $this->payment_group, $credentials );

					Viveum_General_Models::viveum_save_db_registered_payment( $registered_payment );
				}
			}

			/**
			 * Save order transaction details
			 *
			 * @param int   $order_id order id.
			 * @param array $payment_result payment result.
			 * @param int   $reference_id reference id.
			 */
			public function viveum_save_transactions( $order_id, $payment_result, $reference_id ) {
				$this->wc_order = new WC_Order( $order_id );

				$transaction = array();
				$this->viveum_add_customer_note( $this->payment_id, $order_id );
				$additional_info = '';
				if ( 'viveum_dd' === $this->payment_id || 'viveum_ddsaved' === $this->payment_id || 'viveum_easycredit' === $this->payment_id ) {
					$additional_info = $this->viveum_set_serialize_add_info();
				}

				if ( empty( $payment_result['paymentBrand'] ) ) {
					$payment_result['paymentBrand'] = $this->get_payment_brand();
				}

				$total = $this->get_order_total();
				if ( !$total )
					$total = floatval( $payment_result['amount'] );

				$transaction['order_id']       = $this->wc_order->get_order_number();
				if ( $this->has_subscription( $order_id ) ) {
					$transaction['payment_type']   = "DB";
				} else {
					$transaction['payment_type']   = $this->get_payment_type();
				}
				$transaction['reference_id']   = $reference_id;
				$transaction['payment_brand']  = $payment_result['paymentBrand'];
				$transaction['transaction_id'] = $payment_result['merchantTransactionId'];
				$transaction['payment_id']     = $this->payment_id;
				$transaction['payment_status'] = $payment_result['payment_status'];
				$transaction['amount']         = $total;
				$transaction['currency']       = get_woocommerce_currency();
				$transaction['customer_id']    = ( $this->wc_order->get_user_id() ) ? $this->wc_order->get_user_id() : 0;
				$transaction['short_id']       = $this->viveum_get_short_id_from_descriptor( $payment_result['descriptor'] );
				if($this->has_subscription( $order_id )){
					$subscription_additional_info = array(
						"registrationId" => $payment_result['registrationId'],
						"initialTransactionId" => 
						isset($payment_result['standingInstruction']['initialTransactionId'])?$payment_result['standingInstruction']['initialTransactionId']:''
					);
					$additional_info = maybe_serialize($subscription_additional_info);

				}
				// echo "<pre>", print_r($payment_result, 1), "</pre>";die();
				Viveum_General_Models::viveum_save_db_transaction( $transaction, $additional_info );
			}

			/**
			 * Get short id from descriptor of PAY.ON
			 * 
			 * @param string $descriptor descriptor
			 * @return string short ID
			 */
			private function viveum_get_short_id_from_descriptor( $descriptor ) {
				Viveum_General_Functions::add_log( 'viveum_get_short_id_from_descriptor() descriptor', $descriptor );
				$descriptors = explode( ' ', $descriptor );
				$short_id    = $descriptors[0];

				Viveum_General_Functions::add_log( 'viveum_get_short_id_from_descriptor() short_id', $short_id );
				return $short_id;
			}

			/**
			 * Add order notes
			 * add order notes ( database : wp_comment ) if change payment status at backend
			 *
			 * @param int   $order_id order id.
			 * @param array $status_name status name.
			 */
			private function viveum_add_order_notes( $order_id, $status_name ) {

				$user = get_user_by( 'id', get_current_user_id() );
				/**
				 * $timezone = new DateTimeZone( wc_timezone_string() );
				 * date_default_timezone_set( wc_timezone_string() );
				 */

				$comments['order_id'] = $order_id;
				$comments['author']   = $user->display_name;
				$comments['email']    = $user->user_email;
				$comments['content']  = 'Order status changed from ' . $status_name['original'] . ' to ' . $status_name['new'] . '.';

				Viveum_General_Models::viveum_add_db_order_notes( $comments );

			}

			/**
			 * Get merchants info for version tracker
			 *
			 * @return array
			 */
			public function viveum_get_merchant_info() {
				return array_merge(
					$this->viveum_get_general_merchant_info(),
					$this->viveum_get_credit_card_merchant_info()
				);
			}

			/**
			 * Get general merchants info for version tracker
			 *
			 * @return array
			 */
			protected function viveum_get_general_merchant_info() {
				$merchant['transaction_mode'] = $this->settings['server_mode'];
				$merchant['ip_address']       = isset( $_SERVER['SERVER_ADDR'] ) ? sanitize_text_field( wp_unslash( $_SERVER['SERVER_ADDR'] ) ) : ''; // input var okay.
				$merchant['shop_version']     = WC()->version;
				$merchant['plugin_version']   = constant( 'VIVEUM_VERSION' );
				$merchant['client']           = 'Viveum';
				$merchant['merchant_id']      = get_option( 'viveum_general_merchant_no' );
				$merchant['shop_system']      = 'WOOCOMMERCE';
				$merchant['email']            = get_option( 'viveum_general_merchant_email' );
				$merchant['shop_url']         = get_option( 'viveum_general_shop_url' );

				return $merchant;
			}

			/**
			 * Get credit card merchants info for version tracker
			 *
			 * @return array
			 */
			protected function viveum_get_credit_card_merchant_info() {
				$merchant = array();

				if ( 'viveum_cc' === $this->payment_id || 'viveum_ccsaved' === $this->payment_id ) {
					$merchant['merchant_location'] = get_option( 'viveum_general_merchant_location' );
				}

				return $merchant;
			}

			/**
			 * Get error details from easycredit
			 *
			 * @param array $payment_response payment response.
			 * @return array
			 */
			protected function get_easycredit_error_detail( $payment_response ) {
				$error_results = $this->explode_by_multi_delimiter(
					array( '{', '}' ),
					$payment_response['resultDetails']['Error']
				);
				$error_results = explode( ', ', $error_results[1] );

				foreach ( $error_results as $error_result ) {
					$error_result_value                                 = explode( '=', $error_result );
					$easy_credit_error_detail[ $error_result_value[0] ] = count( $error_result_value ) > 1 ? trim( $error_result_value[1], "'" ) : trim( $error_result_value[0], "'" );
				}

				return $easy_credit_error_detail;
			}

			/**
			 * Explode string with multi delimiter
			 *
			 * @param array  $delimiters delimiters.
			 * @param string $string string to be explode.
			 * @return array
			 */
			protected function explode_by_multi_delimiter( $delimiters, $string ) {
				$string          = str_replace( $delimiters, $delimiters[0], $string );
				$exploded_string = explode( $delimiters[0], $string );

				return $exploded_string;
			}

			public function set_without_redirect( $value ) {
				$this->without_redirect = $value;
			}

			public function get_payment_id() {
				return $this->payment_id;
			}

		} // End of class Viveum_Payment_Gateway
	}// End if().

	/**
	 * Add Viveum gateway to WooCommerce
	 *
	 * @access public
	 * @param array $methods methods.
	 * @return array $methods
	 */
	function add_viveumecomerce_payments( $methods ) {
		$methods[] = 'Gateway_Viveum_CC';
		$methods[] = 'Gateway_Viveum_CCSaved';
		$methods[] = 'Gateway_Viveum_DD';
		$methods[] = 'Gateway_Viveum_DDSaved';
		$methods[] = 'Gateway_Viveum_Giropay';
		// $methods[] = 'Gateway_Viveum_Paydirekt';
		$methods[] = 'Gateway_Viveum_Paypal';
		$methods[] = 'Gateway_Viveum_PaypalSaved';
		// $methods[] = 'Gateway_Viveum_Klarnapaylater';
		// $methods[] = 'Gateway_Viveum_Klarnasliceit';
		$methods[] = 'Gateway_Viveum_Klarnaobt';
		// $methods[] = 'Gateway_Viveum_Easycredit';
		$methods[] = 'Gateway_Viveum_Googlepay';
		$methods[] = 'Gateway_Viveum_Applepay';
		$methods[] = 'Gateway_Viveum_EPS';

		return $methods;
	}

	add_filter( 'woocommerce_payment_gateways', 'add_viveumecomerce_payments' );
	foreach ( glob( dirname( __FILE__ ) . '/includes/gateways/**.php' ) as $filename ) {
		include_once $filename;
	}
}


add_action( 'wp_ajax_my_action', 'get_ajax_registered_payment' );
/**
 * Get Registered Payment ( AJAX )
 * Handle ajax request from add payment methods at backend order
 * Echo json
 */
function get_ajax_registered_payment() {
	global $wpdb;

	$payment_group = '';
	$user_id       = Viveum_General_Functions::viveum_get_request_value( 'user_id' );
	$payment_id    = Viveum_General_Functions::viveum_get_request_value( 'payment_id' );

	if ( 'viveum_ccsaved' === $payment_id ) {
		$payment_group = 'CC';
	} elseif ( 'viveum_ddsaved' === $payment_id ) {
		$payment_group = 'DD';
	} elseif ( 'viveum_paypalsaved' === $payment_id ) {
		$payment_group = 'PAYPAL';
	}

	if ( isset( $payment_group ) ) {
		$registered_payment = $wpdb->get_results( $wpdb->prepare( "SELECT  * FROM {$wpdb->prefix}viveum_payment_recurring WHERE cust_id = %d AND payment_group = %s", $user_id, $payment_group ), ARRAY_A ); // db call ok; no-cache ok.
		echo wp_json_encode( $registered_payment );
	}

	wp_die(); // this is required to terminate immediately and return a proper response.
}
