<?php
/**
 * Viveum Webhook Receiver
 *
 * The class is used for receive payload from ACI PAY.ON 
 * Copyright (c) Viveum
 *
 * @class       Viveum_Webhook_Receiver
 * @package     Viveum/Classes
 * @located at  /includes/api/
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

class Viveum_Webhook_Receiver {
	private $payment_id;
	private $gateway;
	private $wc_order;
	private $supported_payment_type = ['DB', 'PA'];

	public function post() {
		Viveum_General_Functions::add_log("Viveum_Webhook_Receiver::post headers", getallheaders());
		Viveum_General_Functions::add_log("Viveum_Webhook_Receiver::post viveum_get_json_post", Viveum_General_Functions::viveum_get_json_post());

		$payload = $this->decrypt_payload();
		
		Viveum_General_Functions::add_log("Viveum_Webhook_Receiver::post payload", $payload);

		// sleep 10 second so it doesn't overlap with actual redirection
		sleep(10);
		
		if ( !$payload ) {
			return "Unable to decrypt";
		}

		return $this->handle_payload_from_payon( $payload );
	}

	private function decrypt_payload() {
		$headers = getallheaders();
		$key_from_configuration = get_option('viveum_secret_for_encryption');
		$iv_from_http_header = $headers['X-Initialization-Vector'];
		$auth_tag_from_http_header = $headers['X-Authentication-Tag'];
		$http_body = Viveum_General_Functions::viveum_get_json_post("encryptedBody");

		try {
			if ( Viveum_General_Functions::is_php_version_lower_than_71() && Viveum_General_Functions::is_sodium_crypto_aead_accessible() ) {
				$key = hex2bin($key_from_configuration);
				$iv = hex2bin($iv_from_http_header);
				$cipher_text = hex2bin($http_body . $auth_tag_from_http_header);
				
				$result = \Sodium\crypto_aead_aes256gcm_decrypt($cipher_text, NULL, $iv, $key);

			} else {
				$key = hex2bin($key_from_configuration);
				$iv = hex2bin($iv_from_http_header);
				$auth_tag = hex2bin($auth_tag_from_http_header);
				$cipher_text = hex2bin($http_body);
				
				$result = openssl_decrypt($cipher_text, "aes-256-gcm", $key, OPENSSL_RAW_DATA, $iv, $auth_tag);
			}

			if ( gettype( $result ) == 'string' ) {
				$result = json_decode( $result, true );
			}

			return $result;

		} catch (Exception $e) {
			Viveum_General_Functions::add_log("Viveum_Webhook_Receiver::decrypt_payload error ", $e->getMessage());
			return false;
		}
	}

	private function handle_payload_from_payon($payload = array()) {
		if ( $payload['type'] !== 'PAYMENT' ) {
			return "type is not supported";
		}

		$payment_result = $payload['payload'];

		if ( ! in_array( $payment_result['paymentType'], $this->supported_payment_type ) ) {
			return "payload.paymentType is not supported";
		}

		$order_id = $payment_result['customParameters']['wc_order_id'];
		$this->payment_id = $payment_result['customParameters']['viveum_payment_id'];

		$this->set_gateway_by_payment_id();
		$this->gateway->set_without_redirect( true );

		$this->wc_order = false;
		try {
			$this->wc_order = new WC_Order( $order_id );

		} catch ( Exception $e ) {
			Viveum_General_Functions::add_log("Viveum_Webhook_Receiver::handle_payload_from_payon error ", $e->getMessage());
			return "Order ID is not found";
		}

		if ( in_array( $payment_result['paymentType'], ['DB', 'PA'] ) ) {
			return $this->handle_payload_for_db_pa_payment_type( $payment_result );
		}
	}

	private function set_gateway_by_payment_id() {
		$methods = apply_filters( 'woocommerce_payment_gateways', array() );

		foreach ( $methods as $method ) {
			$gateway = new $method();
			if ( $gateway->get_payment_id() === $this->payment_id ) {
				$this->gateway = $gateway;
				break;
				return false;
			}
		}
	}

	private function handle_payload_for_db_pa_payment_type( $payment_result ) {
		$result_status = ViveumPaymentCore::get_transaction_result( $payment_result['result']['code'] );
		$payment_status = $this->get_db_pa_payment_status( $payment_result );

		Viveum_General_Functions::add_log( 'Viveum_Webhook_Receiver.handle_payload_for_db_pa_payment_type status by payment result', $payment_status, $this->wc_order->get_id() );

		if ( 'wc-' . $this->wc_order->get_status() === $payment_status ) {
			Viveum_General_Functions::add_log( 'Viveum_Webhook_Receiver.handle_payload_for_db_pa_payment_type is already ', $payment_status, $this->wc_order->get_id() );
			return "Status is already " . $payment_status;
		}

		$payment_result['payment_status'] = $payment_status;

		if ( $payment_status === 'wc-failed' ) {
			return $this->handle_failed_db_pa_payment( $payment_result );

		} else {
			return $this->handle_success_or_pending_db_pa_payment( $payment_result );
		}

	}

	private function get_db_pa_payment_status( $payment_result ) {
		$is_success_review = ViveumPaymentCore::is_success_review( $payment_result['result']['code'] );
		$result_status = ViveumPaymentCore::get_transaction_result( $payment_result['result']['code'] );

		if ( $is_success_review ) {
			$payment_status = 'wc-in-review';
		} elseif ( $result_status === "PD" ) {
			$payment_status = 'wc-pending';
		} elseif ( $result_status === "NOK" ) {
			$payment_status = 'wc-failed';
		} elseif ( $result_status === "ACK" ) {
			if ( $payment_result['paymentType'] === 'PA' ) {
				$payment_status = 'wc-pre-authorization';
			} else {
				$payment_status = 'wc-payment-accepted';
			}
		} else {
			Viveum_General_Functions::add_log("Viveum_Webhook_Receiver.get_db_pa_payment_status RESULT STATUS IS NOT FOUND");
			$payment_status = false;
		}

		return $payment_status;
	}

	private function handle_failed_db_pa_payment( $payment_result ) {
		$this->gateway->viveum_save_transactions( $this->wc_order->get_id(), $payment_result, $payment_result['id'] );

		Viveum_General_Functions::add_log("Viveum_Webhook_Receiver.handle_failed_db_pa_payment saved transaction.", '', $this->wc_order->get_id());

		$error_identifier = 'ERROR_UNKNOWN';
		$this->gateway->viveum_do_error_payment( $this->wc_order->get_id(), $payment_result['payment_status'], $error_identifier );

		Viveum_General_Functions::add_log("Viveum_Webhook_Receiver.handle_failed_db_pa_payment status changed.", '', $this->wc_order->get_id());

		return "Successfully save payment status to " . $payment_result['payment_status'];
	}

	private function handle_success_or_pending_db_pa_payment( $payment_result ) {
		$result_status = ViveumPaymentCore::get_transaction_result( $payment_result['result']['code'] );
		$payment_status = $payment_result['payment_status'];

		if ( 'PD' === $result_status && ! $this->gateway->is_payment_method_allowed_to_continue( $payment_result['paymentBrand'] ) ) {
			$this->gateway->viveum_cancel_payment( $this->wc_order->get_id() );
			$payment_result['payment_status'] = 'wc-cancelled';

			Viveum_General_Functions::add_log("Viveum_Webhook_Receiver.handle_success_or_pending_db_pa_payment cancel payment", false, $this->wc_order->get_id());

		} else {
			if ( $this->gateway->is_recurring() ) {
				$account = Viveum_General_Functions::viveum_get_account_by_result( $this->payment_id, $payment_result );
				if ( 'viveum_paypalsaved' === $this->payment_id ) {
					$registration_id                              = $payment_result['id'];
					$payment_brand                                = $payment_result['paymentBrand'];
					$payment_result                               = $this->gateway->viveum_debit_paypal_recurring( $this->wc_order->get_id(), $registration_id );
					$payment_result['registrationId'] = $registration_id;
					$payment_result['paymentBrand']   = $payment_brand;
					$payment_result['payment_status'] = $payment_status;
				}

				$this->gateway->viveum_save_recurring_payment( $this->wc_order->get_id(), $payment_result, $account );

				Viveum_General_Functions::add_log("Viveum_Webhook_Receiver.handle_success_or_pending_db_pa_payment saved recurring account.", false, $this->wc_order->get_id());
			}

			$this->gateway->viveum_do_success_payment( $this->wc_order->get_id(), $payment_result );

			Viveum_General_Functions::add_log("Viveum_Webhook_Receiver.handle_success_or_pending_db_pa_payment success payment.", false, $this->wc_order->get_id());
		}

		return "Successfully save payment status to " . $payment_result['payment_status'];
	}
}

$api = new Viveum_Webhook_Receiver();
register_rest_route(
	VIVEUM_API_NAMESPACE,
	'/webhook-receiver',
	array(
		'methods' => 'POST',
		'callback' => array(
			$api,
			'post'
		)
	),
	true
);