<?php
/**
 * Viveum General Functions
 *
 * General functions available on both the front-end and admin.
 * Copyright (c) Viveum
 *
 * @class       Viveum_General_Functions
 * @package     Viveum/Classes
 * @located at  /includes/
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * General functions available on both the front-end and admin.
 */
class Viveum_General_Functions {
	/**
	 * Percent for discount or  tax
	 *
	 * @var int $percent percent.
	 */
	public static $percent = 100;

	/**
	 * Get the plugin url.
	 *
	 * @return string
	 */
	public static function viveum_get_plugin_url() {

		return untrailingslashit( plugins_url( '/', VIVEUM_PLUGIN_FILE ) );
	}

	/**
	 * Return the WordPress language
	 *
	 * @return string
	 */
	public static function viveum_get_shop_language() {

		return ( substr( get_bloginfo( 'language' ), 0, 2 ) === 'de' ) ? 'de' : 'en' ;
	}

	/**
	 * Get payment gateway instance by id
	 *
	 * @param string $id id.
	 * @return object
	 */
	public static function viveum_get_payment_gateway_by_id( $id ) {
		$method = ucfirst( str_replace( 'viveum_', '', $id ) );
		$method = str_replace( 'Cc', 'CC', $method );
		$method = str_replace( 'Dd', 'DD', $method );
		$method = str_replace( 'saved', 'Saved', $method );

		$class_name = 'Gateway_Viveum_' . $method;
		$payment_gateway = new $class_name();

		return $payment_gateway;
	}

	/**
	 * Get Customer IP.
	 *
	 * @return string
	 */
	public static function get_customer_ip() {
		$server_remote_addr = '';
		$server = $_SERVER; // input var okay.
		if ( isset( $server['REMOTE_ADDR'] ) ) {
			$server_remote_addr = sanitize_text_field( wp_unslash( $server['REMOTE_ADDR'] ) );
		}

		if ( '::1' === $server_remote_addr ) {
			$customer_ip = '127.0.0.1';
		} else {
			$customer_ip = $server_remote_addr;
		}

		return $customer_ip;
	}

	/**
	 * Check viveum is version tracker active
	 *
	 * @return bool
	 */
	public static function viveum_is_version_tracker_active() {

		return get_option( 'viveum_general_version_tracker' );
	}

	/**
	 * Get Recurring Payment
	 *
	 * @return array
	 */
	public static function viveum_get_recurring_payment() {
		$payments = array( 'viveum_ccsaved', 'viveum_ddsaved', 'viveum_paypalsaved' );

		return $payments;
	}

	/**
	 * Get Payment Form
	 *
	 * @param string $payment_id payment id.
	 * @return string
	 */
	public static function viveum_get_payment_form( $payment_id ) {

		switch ( $payment_id ) {
			case 'viveum_paypal':
			case 'viveum_paydirekt':
				$payment_form = 'redirect';
				break;
			case 'viveum_easycredit':
				$payment_form = 'servertoserver';
				break;
			case 'viveum_paypalsaved':
				$payment_form = 'form_paypalsaved';
				break;
			default:
				$payment_form = 'form';
		}

		return $payment_form;
	}

	/**
	 * Get Payment Id By Group
	 *
	 * @param string $payment_group payment group.
	 * @return string
	 */
	public static function viveum_get_payment_id_by_group( $payment_group ) {

		switch ( $payment_group ) {
			case 'CC':
				$payment_id = 'viveum_ccsaved';
				break;
			case 'DD':
				$payment_id = 'viveum_ddsaved';
				break;
			case 'PAYPAL':
				$payment_id = 'viveum_paypalsaved';
				break;
			default:
				$payment_id = false;
		}

		return $payment_id;
	}

	/**
	 * Is recuring
	 *
	 * @param string $payment_id payment id.
	 * @return bool
	 */
	public static function is_recurring( $payment_id ) {
		switch ( $payment_id ) {
			case 'viveum_ccsaved':
			case 'viveum_ddsaved':
			case 'viveum_paypalsaved':
				return true;
			default:
				return false;
		}
	}

	/**
	 * Get Account Card Info from gateway
	 *
	 * @param string $payment_id payment id.
	 * @param array  $payment_result payment result.
	 * @return array
	 */
	public static function viveum_get_account_by_result( $payment_id, $payment_result ) {

		switch ( $payment_id ) {
			case 'viveum_ccsaved':
				$account_card['holder']         = $payment_result['card']['holder'];
				$account_card['last_4_digits']  = $payment_result['card']['last4Digits'];
				$account_card['expiry_month']   = $payment_result['card']['expiryMonth'];
				$account_card['expiry_year']    = $payment_result['card']['expiryYear'];
				$account_card['email']          = '';
				break;
			case 'viveum_ddsaved':
				$account_card['holder']         = $payment_result['bankAccount']['holder'];
				$account_card['last_4_digits']  = substr( $payment_result['bankAccount']['iban'],-4 );
				$account_card['expiry_month']   = '';
				$account_card['expiry_year']    = '';
				$account_card['email']          = '';
				break;
			case 'viveum_paypalsaved':
				$account_card['holder']         = $payment_result['virtualAccount']['holder'];
				$account_card['last_4_digits']  = '';
				$account_card['expiry_month']   = '';
				$account_card['expiry_year']    = '';
				$account_card['email']          = $payment_result['virtualAccount']['accountId'];
				break;
		}

		return $account_card;

	}

	/**
	 * Get Payment Price with Tax and Discount.
	 *
	 * @param array $cart cart.
	 * @return float
	 */
	public static function viveum_get_payment_price_with_tax_and_discount( $cart ) {
		global $woocommerce;
		$wc_cart = $woocommerce->cart;

		$is_prices_include_tax = get_option( 'woocommerce_prices_include_tax' );
		$price = $cart['data']->get_price();
		$price_with_discount = $price;

		if ( 'no' === $is_prices_include_tax ) {
			$tax = $cart['line_tax'] / $cart['quantity'];
			$price_with_discount_and_tax = $price + $tax;
		} else {
			$price_with_discount_and_tax = $price;
		}

		return ViveumPaymentCore::set_number_format( $price_with_discount_and_tax );
	}

	/**
	 * Get Payment Discount from Cart Order.
	 *
	 * @param array $cart cart.
	 * @return float
	 */
	public static function viveum_get_payment_discount_in_percent( $cart ) {
		$product_detail = Viveum_General_Models::viveum_get_db_product_detail( $cart['data']->get_id() );

		$regular_price = $product_detail['_regular_price'];
		$sale_price = $cart['data']->get_price();

		if ( $regular_price !== $sale_price ) {
			$discount = $regular_price - $sale_price;
			$discount_percent = ( $discount / $regular_price ) * self::$percent;

			return ViveumPaymentCore::set_number_format( $discount_percent );
		}

		return false;
	}

	/**
	 * Get Payment Tax from Cart Order.
	 *
	 * @param array $cart cart.
	 * @return float
	 */
	public static function viveum_get_payment_tax_in_percent( $cart ) {
		$is_enable_tax = get_option( 'woocommerce_calc_taxes' );
		$product_detail = Viveum_General_Models::viveum_get_db_product_detail( $cart['data']->get_id() );
		if ( isset( $product_detail['_tax_status'] ) ) {
			$is_enable_tax_product = $product_detail['_tax_status'];
		}
		if ( isset( $is_enable_tax_product ) && 'yes' === $is_enable_tax && 'taxable' === $is_enable_tax_product ) {
			$tax_precent = ( $cart['line_tax'] / $cart['line_total'] ) * self::$percent;
			return ViveumPaymentCore::set_number_format( $tax_precent );
		}

		return false;
	}

	/**
	 * Get Payment Tax from Cart Order.
	 *
	 * @param string $gender gender.
	 * @return string
	 */
	public static function viveum_get_initial_gender( $gender ) {
		switch ( $gender ) {
			case 'Male':
				$initial_gender = 'M';
				break;
			case 'Female':
				$initial_gender = 'F';
				break;
			default:
				$initial_gender = '';
				break;
		}

		return $initial_gender;
	}

	/**
	 * Translate Error Identifier.
	 *
	 * @param string $error_identifier error identifier.
	 * @return string
	 */
	public static function viveum_translate_error_identifier( $error_identifier ) {
		switch ( $error_identifier ) {
			case 'ERROR_CC_ACCOUNT':
				$error_translate = __( 'The account holder entered does not match your name. Please use an account that is registered on your name.', 'wc-viveum' );
				break;
			case 'ERROR_CC_INVALIDDATA':
				$error_translate = __( 'Unfortunately, the card/account data you entered was not correct. Please try again.', 'wc-viveum' );
				break;
			case 'ERROR_CC_BLACKLIST':
				$error_translate = __( 'Unfortunately, the credit card you entered can not be accepted. Please choose a different card or payment method.', 'wc-viveum' );
				break;
			case 'ERROR_CC_DECLINED_CARD':
				$error_translate = __( 'Unfortunately, the credit card you entered can not be accepted. Please choose a different card or payment method.', 'wc-viveum' );
				break;
			case 'ERROR_CC_EXPIRED':
				$error_translate = __( 'Unfortunately, the credit card you entered can not be accepted. Please choose a different card or payment method.', 'wc-viveum' );
				break;
			case 'ERROR_CC_INVALIDCVV':
				$error_translate = __( 'Unfortunately, the CVV/CVC you entered is not correct. Please try again.', 'wc-viveum' );
				break;
			case 'ERROR_CC_EXPIRY':
				$error_translate = __( 'Unfortunately, the expiration date you entered is not correct. Please try again.', 'wc-viveum' );
				break;
			case 'ERROR_CC_LIMIT_EXCEED':
				$error_translate = __( 'Unfortunately, the limit of your credit card is exceeded. Please choose a different card or payment method.', 'wc-viveum' );
				break;
			case 'ERROR_CC_3DAUTH':
				$error_translate = __( 'Unfortunately, the password you entered was not correct. Please try again.', 'wc-viveum' );
				break;
			case 'ERROR_CC_3DERROR':
				$error_translate = __( 'Unfortunately, there has been an error while processing your request. Please try again.', 'wc-viveum' );
				break;
			case 'ERROR_CC_NOBRAND':
				$error_translate = __( 'Unfortunately, there has been an error while processing your request. Please try again.', 'wc-viveum' );
				break;
			case 'ERROR_GENERAL_LIMIT_AMOUNT':
				$error_translate = __( 'Unfortunately, your credit limit is exceeded. Please choose a different card or payment method.', 'wc-viveum' );
				break;
			case 'ERROR_GENERAL_LIMIT_TRANSACTIONS':
				$error_translate = __( 'Unfortunately, your limit of transaction is exceeded. Please try again later.', 'wc-viveum' );
				break;
			case 'ERROR_CC_DECLINED_AUTH':
				$error_translate = __( 'Unfortunately, your transaction has failed. Please choose a different card or payment method.', 'wc-viveum' );
				break;
			case 'ERROR_GENERAL_DECLINED_RISK':
				$error_translate = __( 'Unfortunately, your transaction has failed. Please choose a different card or payment method.', 'wc-viveum' );
				break;
			case 'ERROR_CC_ADDRESS':
				$error_translate = __( 'We are sorry. We could no accept your card as its origin does not match your address.', 'wc-viveum' );
				break;
			case 'ERROR_GENERAL_CANCEL':
				$error_translate = __( 'You cancelled the payment prior to its execution. Please try again.', 'wc-viveum' );
				break;
			case 'ERROR_CC_RECURRING':
				$error_translate = __( 'Recurring transactions have been deactivated for this credit card. Please choose a different card or payment method.', 'wc-viveum' );
				break;
			case 'ERROR_CC_REPEATED':
				$error_translate = __( 'Unfortunately, your transaction has been declined due to invalid data. Please choose a different card or payment method.', 'wc-viveum' );
				break;
			case 'ERROR_GENERAL_ADDRESS':
				$error_translate = __( 'Unfortunately, your transaction has failed. Please check the personal data you entered.', 'wc-viveum' );
				break;
			case 'ERROR_GENERAL_BLACKLIST':
				$error_translate = __( 'The chosen payment method is not available at the moment. Please choose a different card or payment method.', 'wc-viveum' );
				break;
			case 'ERROR_GENERAL_GENERAL':
				$error_translate = __( 'Unfortunately, your transaction has failed. Please try again.', 'wc-viveum' );
				break;
			case 'ERROR_GENERAL_TIMEOUT':
				$error_translate = __( 'Unfortunately, your transaction has failed. Please try again.', 'wc-viveum' );
				break;
			case 'ERROR_GIRO_NOSUPPORT':
				$error_translate = __( 'Giropay is not supported for this transaction. Please choose a different payment method.', 'wc-viveum' );
				break;
			case 'ERROR_CAPTURE_BACKEND':
				$error_translate = __( 'Transaction can not be captured.', 'wc-viveum' );
				break;
			case 'ERROR_REORDER_BACKEND':
				$error_translate = __( 'Card holder has advised his bank to stop this recurring payment.', 'wc-viveum' );
				break;
			case 'ERROR_REFUND_BACKEND':
				$error_translate = __( 'Transaction can not be refunded or reversed.', 'wc-viveum' );
				break;
			case 'ERROR_RECEIPT_BACKEND':
				$error_translate = __( 'Receipt can not be performed', 'wc-viveum' );
				break;
			case 'ERROR_ADDRESS_PHONE':
				$error_translate = __( 'Unfortunately, your transaction has failed. Please enter a valid telephone number.', 'wc-viveum' );
				break;
			case 'ERROR_GENERAL_NORESPONSE':
				$error_translate = __( 'Unfortunately, the confirmation of your payment failed. Please contact your merchant for clarification.', 'wc-viveum' );
				break;
			case 'ERROR_WRONG_DOB':
				$error_translate = __( 'Wrong date of birth format', 'wc-viveum' );
				break;
			case 'ERROR_EASYCREDIT_PARAMETER_DOB':
				$error_translate = __( 'Please enter your date of birth in order to use easyCredit.', 'wc-viveum' );
				break;
			case 'ERROR_MESSAGE_EASYCREDIT_AMOUNT_NOTALLOWED':
				$error_translate = __( 'The financing amount is outside the permitted amounts (200 - 5,000 EUR)', 'wc-viveum' );
				break;
			case 'ERROR_EASYCREDIT_DOB_LOWER_THAN_TODAY':
				$error_translate = __( 'Date of birth should not be set in the future', 'wc-viveum' );
				break;
			case 'ERROR_EASYCREDIT_BILLING_NOTEQUAL_SHIPPING':
				$error_translate = __( 'In order to be able to pay with easyCredit, the delivery address must match the invoice address.', 'wc-viveum' );
				break;
			case 'ERROR_EASYCREDIT_NO_SHIPPING_METHOD':
				$error_translate = __( 'Please select a shipping method to use easyCredit', 'wc-viveum' );
				break;
			case 'ERROR_PARAMETER_GENDER':
				$error_translate = __( 'Please enter your gender to make payment.', 'wc-viveum' );
				break;
			case 'ERROR_MESSAGE_EASYCREDIT_PARAMETER_GENDER':
				$error_translate = __( 'Please enter your gender to make payment with easyCredit.', 'wc-viveum' );
				break;
			case 'ERROR_GENERAL_REDIRECT':
				$error_translate = __( 'Error before redirect', 'wc-viveum' );
				break;
			case 'ERROR_MC_ADD':
				$error_translate = __( 'We are sorry. Your attempt to save your payment information was not successful.', 'wc-viveum' );
				break;
			case 'ERROR_MC_UPDATE':
				$error_translate = __( 'We are sorry. Your attempt to update your payment information was not successful.', 'wc-viveum' );
				break;
			case 'ERROR_MC_DELETE':
				$error_translate = __( 'We are sorry. Your attempt to delete your payment information was not successful.', 'wc-viveum' );
				break;
			case 'ERROR_MERCHANT_SSL_CERTIFICATE':
				$error_translate = __( 'SSL certificate problem, please contact the merchant.', 'wc-viveum' );
				break;
			default:
				$error_translate = __( 'Unfortunately, your transaction has failed. Please try again.', 'wc-viveum' );
				break;
		}// End switch().

		return $error_translate;
	}

	/**
	 * Translate viveum_term
	 *
	 * @param string $viveum_term term.
	 * @return string
	 */
	public static function viveum_translate_viveum_term( $viveum_term ) {
		switch ( $viveum_term ) {
			case 'VIVEUMECOMMERCE_TT_VERSIONTRACKER':
				$viveum_term_translate = __( 'By enabling Viveum - WooCommerce, you accept to share your IP, email address, etc with Viveum. If you wish not to share those information, you can change the setting via', 'wc-viveum' );
				break;
			case 'VIVEUMECOMMERCE_BACKEND_BT_ADMIN':
				$viveum_term_translate = __( 'Admin', 'wc-viveum' );
				break;
		}
		return $viveum_term_translate;
	}

	/**
	 * Translate Additional Information Name
	 *
	 * @param string $info_name info name.
	 * @return string
	 */
	public static function viveum_translate_additional_information( $info_name ) {
		switch ( $info_name ) {
			case 'FRONTEND_EASYCREDIT_INTEREST':
				$additional_info_name = __( 'Sum of Interest', 'wc-viveum' );
				break;
			case 'FRONTEND_EASYCREDIT_TOTAL':
				$additional_info_name = __( 'Order Total', 'wc-viveum' );
				break;
			default:
				$additional_info_name = $info_name;
				break;
		}

		return $additional_info_name;
	}

	/**
	 * Get _REQUEST value
	 *
	 * @param string $key key.
	 * @param string $default default.
	 * @return value
	 */
	public static function viveum_get_request_value( $key, $default = false ) {
		if ( isset( $_REQUEST[ $key ] ) ) {// input var okay.
			return sanitize_text_field( wp_unslash( $_REQUEST[ $key ] ) ); // input var okay.
		}
		return $default;
	}

	/**
	 * Get _GET value
	 *
	 * @param string $key key.
	 * @param string $default default.
	 * @return value
	 */
	public static function viveum_get_query_string( $key, $default = false ) {
		if ( isset( $_GET[ $key ] ) ) {// input var okay.
			return sanitize_text_field( wp_unslash( $_GET[ $key ] ) ); // input var okay.
		}
		return $default;
	}

	/**
	 * Get JSON value
	 *
	 * @param string $key key.
	 * @param string $default default.
	 * @return string | array | boolean
	 */
	public static function viveum_get_json_post( $key = false, $default = false ) {
		$json = file_get_contents('php://input');
		if ( $json ) {
			$json = json_decode($json, true);
		} else {
			return $default;
		}
		if ( !$key ) {
			return $json;
		}
		if ( isset( $json[ $key ] ) ) {// input var okay.
			return sanitize_text_field( wp_unslash( $json[ $key ] ) ); // input var okay.
		}
		return $default;
	}

	/**
	 * Include template
	 *
	 * @param string $template_file_path template file path (templates/template.php).
	 * @param array  $args variable to include in template.
	 */
	public static function viveum_include_template( $template_file_path, $args = array() ) {
		if ( function_exists( 'wc_get_template' ) ) {
			$template = pathinfo( $template_file_path );
			$template_path = $template['dirname'] . '/';
			$template_file = $template['basename'];
			wc_get_template( $template_file,
				$args,
				$template_path,
				$template_path
			);
		} else {
			foreach ( $args as $key => $value ) {
				$$key = $value;
			}
			include( $template_file_path );
		}
	}

	/**
	 * Validate birth of date
	 *
	 * @param  string $dob birth of date.
	 * @return string|boolean
	 */
	public static function viveum_is_date_of_birth_valid( $dob ) {
		if ( ! empty( $dob ) ) {

			$customer_date_of_birth = explode( '.', $dob );

			if ( ! isset( $customer_date_of_birth[0] ) ) {
				return false;
			}
			if ( ! isset( $customer_date_of_birth[1] ) ) {
				return false;
			}
			if ( ! isset( $customer_date_of_birth[2] ) ) {
				return false;
			}

			$day = (int) $customer_date_of_birth[0];
			$month = (int) $customer_date_of_birth[1];
			$year = (int) $customer_date_of_birth[2];

			if ( $year < 1900 ) {
				return false;
			}

			if ( $month < 1 || $month > 12 ) {
				return false;
			}

			if ( $day < 1 || $day > 31 ) {
				return false;
			}

			$valid = checkdate( $month, $day, $year );

			if ( $valid ) {
				return true;
			}
		}// End if().
		return false;
	}

	/**
	 * Validate future birth of date
	 *
	 * @param  string $dob birth of date.
	 * @return string|boolean
	 */
	public static function viveum_is_date_of_birth_lower_than_today( $dob ) {
		$customer_date_of_birth = strtotime( $dob );
		$today = strtotime( date( 'd-m-Y' ) );

		if ( $customer_date_of_birth >= $today ) {
			return false;
		}
		return true;
	}

	/**
	 * Is woocommerce version greater than
	 *
	 * @param string $version woocommerce version.
	 * @return boolean
	 */
	public static function is_version_greater_than( $version ) {
		if ( class_exists( 'WooCommerce' ) ) {
			global $woocommerce;
			if ( version_compare( $woocommerce->version, $version, '>=' ) ) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Get wc customer property value
	 *
	 * @param  string $property property of class WC_Order.
	 * @return mixed
	 */
	protected static function get_wc_customer_property_value( $property ) {
		global $woocommerce;

		$customer = $woocommerce->customer;
		$function = $property;
		return $customer->$function();
	}

	/**
	 * Check if billing address and shipping addres are equal
	 *
	 * @return boolean
	 */
	public static function viveum_is_address_billing_equal_shipping() {
		global $woocommerce;

		$customer = $woocommerce->customer;

		if ( self::is_version_greater_than( '3.0' ) ) {
			$billing['street']        = self::get_wc_customer_property_value( 'get_billing_address_1' ) . ', ' . self::get_wc_customer_property_value( 'get_billing_address_2' );
			$billing['city']          = self::get_wc_customer_property_value( 'get_billing_city' );
			$billing['zip']           = self::get_wc_customer_property_value( 'get_billing_postcode' );
			$billing['country_code']  = self::get_wc_customer_property_value( 'get_billing_country' );
			$shipping['street']        = self::get_wc_customer_property_value( 'get_shipping_address_1' ) . ', ' . self::get_wc_customer_property_value( 'get_shipping_address_2' );
			$shipping['city']          = self::get_wc_customer_property_value( 'get_shipping_city' );
			$shipping['zip']           = self::get_wc_customer_property_value( 'get_shipping_postcode' );
			$shipping['country_code']  = self::get_wc_customer_property_value( 'get_shipping_country' );
		} else {
			$billing['street']        = self::get_wc_customer_property_value( 'get_address' ) . ', ' . self::get_wc_customer_property_value( 'get_address_2' );
			$billing['city']          = self::get_wc_customer_property_value( 'get_city' );
			$billing['zip']           = self::get_wc_customer_property_value( 'get_postcode' );
			$billing['country_code']  = self::get_wc_customer_property_value( 'get_country' );
			$shipping['street']        = self::get_wc_customer_property_value( 'get_shipping_address' ) . ', ' . self::get_wc_customer_property_value( 'get_shipping_address_2' );
			$shipping['city']          = self::get_wc_customer_property_value( 'get_shipping_city' );
			$shipping['zip']           = self::get_wc_customer_property_value( 'get_shipping_postcode' );
			$shipping['country_code']  = self::get_wc_customer_property_value( 'get_shipping_country' );
		}

		foreach ( $billing as $i => $bill ) {
			if ( $bill !== $shipping[ $i ] ) {
				return false;
			}
		}
		return true;
	}

	/**
	 * Check if shipping method is chosen
	 *
	 * @return boolean
	 */
	public static function viveum_is_shipping_method_chosen() {
		$chosen_shipping_method = WC()->session->get( 'chosen_shipping_methods' );

		if ( ! is_null( $chosen_shipping_method[0] ) ) {
			return true;
		}

		return false;
	}

	/**
	 * Determines if amount allowed
	 *
	 * @param int $total_amount total amount.
	 * @return boolean
	 */
	public static function viveum_is_amount_allowed( $total_amount ) {

		$is_currency_euro = self::viveum_is_currency_euro();

		if ( $is_currency_euro || $total_amount < 200 || $total_amount > 5000 ) {
			return false;
		}
		return true;
	}

	/**
	 * Determines if currency euro.
	 *
	 * @return boolean
	 */
	public static function viveum_is_currency_euro() {
		$currency = get_woocommerce_currency();

		if ( 'EUR' !== $currency ) {
			return true;
		}
		return false;
	}

	/**
	 * Get customer order count
	 *
	 * @return int count
	 */
	public static function viveum_get_order_count() {
		if ( wc_get_customer_order_count( get_current_user_id() ) > 0 ) {
			return wc_get_customer_order_count( get_current_user_id() );
		}
		return 0;
	}

	/**
	 * Get risk kunden status for easyCredit
	 *
	 * @return string risk kunden
	 */
	public static function viveum_get_risk_kunden_status() {
		if ( self::viveum_get_order_count() > 0 ) {
			return 'BESTANDSKUNDE';
		}
		return 'NEUKUNDE';
	}

	/**
	 * Get customer created date
	 *
	 * @return string|boolean created date
	 */
	public static function viveum_get_customer_created_date() {
		if ( is_user_logged_in() ) {
			$user_data = get_userdata( get_current_user_id() );
			$created_date = strtotime( $user_data->user_registered );

			if ( isset( $user_data->user_registered ) && $created_date > 0 ) {
				return date( 'Y-m-d', $created_date );
			}
		}
		return date( 'Y-m-d' );
	}

	/**
	 * Add log for debugging
	 *
	 * @param string            $message - message.
	 * @param bool|string|array $data - data.
	 * @param bool|int          $order_id - order id.
	 */
	public static function add_log( $message, $data = false, $order_id = false ) {
		$viveum_log = new WC_Logger();
		$logger_handle = 'viveum-' . date( 'Ym' );

		if ( ! $order_id ) {
			$logger_message = $message;
		} elseif ( isset( $order_id ) || '0' !== $order_id ) {
			$wc_order = new WC_Order( $order_id );
			$order_id = $wc_order->get_order_number();

			$logger_message = '[Order Id : ' . $order_id . '] ' . $message;
		}

		if ( $data ) {
			if ( is_object( $data ) ) {
				$logger_message .= " : Xml ( \r\n";
				foreach ( $data->children() as $child ) {
					$logger_message .= '(' . $child->getName() . ') => ' . $child . "\r\n";
				}
				$logger_message .= ')';
			} elseif ( is_array( $data ) ) {
				$logger_message .= ' : ' . print_r( $data , 1 );
			} else {
				$logger_message .= ' : ' . $data;
			}
		}

		$viveum_log->add( $logger_handle, $logger_message . "\r\n" );
	}

	/**
	 * Is php version lower than ?
	 *
	 * @return boolean
	 */
	public static function is_php_version_lower_than_71() {
		return version_compare(phpversion(), "7.1", "<");
	}

	/**
	 * Is \Sodium\crypto_aead_aes256gcm_decrypt accessible ?
	 *
	 * @return boolean
	 */
	public static function is_sodium_crypto_aead_accessible() {
		try {
			if (\Sodium\crypto_aead_aes256gcm_is_available()) {
				return true;
			}
		} catch( Exception $e ) {
			return false;
		}

		return false;
	}
}
