<?php
/**
 * Viveum Plugin Installation Process
 *
 * The class is used for My Payment Information
 * Copyright (c) Viveum
 *
 * @class       Viveum_Payment_Information
 * @package     Viveum/Classes
 * @located at  /includes/myaccount/
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * The class is used for My Payment Information
 */
class Viveum_Payment_Information extends WC_Shortcode_My_Account {

	/**
	 * Recurring
	 *
	 * @var boolean $recurring
	 */
	static $recurring;
	/**
	 * Home url
	 *
	 * @var string $home_url
	 */
	static $home_url;
	/**
	 * Current url
	 *
	 * @var string $current_url
	 */
	static $current_url;
	/**
	 * Clean current url
	 *
	 * @var string $clean_current_url
	 */
	static $clean_current_url;
	/**
	 * Plugin url
	 *
	 * @var string $plugin_url
	 */
	static $plugin_url;
	/**
	 * Plugin path
	 *
	 * @var string $plugin_path
	 */
	static $plugin_path;

	/**
	 * Main function of the class
	 */
	public static function init() {
		global $wp, $wp_query;

		self::$home_url = home_url();
		if ( isset( $wp->request ) ) {
			self::$current_url = home_url( $wp->request ) . '/?';
		} else {
			self::$current_url = get_page_link() . '&';
		}
		self::$clean_current_url = rtrim( self::$current_url, '?' );
		self::$plugin_url = plugins_url( '/', VIVEUM_PLUGIN_FILE );
		self::$plugin_path = dirname( __FILE__ ) . '/../..';

		if ( ! is_user_logged_in() ) {
			wp_safe_redirect( get_permalink( get_option( 'woocommerce_myaccount_page_id' ) ) );
		} else {
			self::$recurring = get_option( 'viveum_general_recurring' );
			if ( self::$recurring ) {
				$section = Viveum_General_Functions::viveum_get_request_value( 'viveum_section' );
				$page = Viveum_General_Functions::viveum_get_query_string( 'viveum_page', '' );
				switch ( $page ) {
					case 'wc-register':
						self::viveum_register_payment( $section );
						break;
					case 'wc-reregister':
						self::viveum_change_payment( $section );
						break;
					case 'wc-deregister':
						self::viveum_delete_payment( $section );
						break;
					case 'wc-response-register':
						self::viveum_response_register( $section, 'register' );
						break;
					case 'wc-response-reregister':
						self::viveum_response_register( $section, 'reregister' );
						break;
					case 'wc-response-deregister':
						self::viveum_response_deregister( $section );
						break;
					case 'wc-default':
						self::viveum_set_default_payment( $section );
						break;
					default:
						self::viveum_get_payment_information_page();
						break;
				}
			} else {
				wp_safe_redirect( get_permalink( get_option( 'woocommerce_myaccount_page_id' ) ) );
			}
		}// End if().
	}

	/**
	 * Load payment information page
	 * List of My Payment Information ( CC, DD & PAYPAL ).
	 */
	private static function viveum_get_payment_information_page() {
		$cc_saved = new Gateway_Viveum_CCSaved;
		$dd_saved = new Gateway_Viveum_DDSaved;
		$paypal_saved = new Gateway_Viveum_PaypalSaved;

		$payment_gateway_recurring = array( $cc_saved, $dd_saved, $paypal_saved );

		$args = array(
			'plugin_url' => self::$plugin_url,
			'recurring' => self::$recurring,
			'current_url' => self::$current_url,
		);
		foreach ( $payment_gateway_recurring as $payment_gateway ) {
			$payment_group = $payment_gateway->get_payment_group();
			$credentials = $payment_gateway->get_credentials();
			$registered_payments[ $payment_group ] = Viveum_General_Models::viveum_get_db_registered_payment( $payment_group, $credentials );

			$enabled = 'is_active_' . strtolower( $payment_group );
			$args[ $enabled ] = 'yes' === $payment_gateway->enabled ? true : false;
		}
		$args['registered_payments'] = $registered_payments;

		wp_enqueue_style( 'viveum_formpayment_style', self::$plugin_url . '/assets/css/formpayment.css', array(), null );
		Viveum_General_Functions::viveum_include_template( self::$plugin_path . '/templates/myaccount/template-payment-information.php', $args );
	}

	/**
	 * Check if payment active
	 *
	 * @param string $payment_id payment id.
	 * @return bool
	 */
	private static function viveum_is_active_payment( $payment_id ) {
		$payment_settings = get_option( 'woocommerce_' . $payment_id . '_settings' );

		if ( 'yes' === $payment_settings['enabled'] ) {
			return true;
		} else {
			return false;
		}
	}


	/**
	 * Set default payment account
	 *
	 * @param string $payment_id payment id.
	 */
	private static function viveum_set_default_payment( $payment_id ) {
		$payment_gateway = Viveum_General_Functions::viveum_get_payment_gateway_by_id( $payment_id );
		$recurring_id = Viveum_General_Functions::viveum_get_request_value( 'id' );

		if ( $recurring_id ) {
			$payment_group = $payment_gateway->get_payment_group();
			$credentials = $payment_gateway->get_credentials();

			$query['field'] = 'payment_default';
			$query['value'] = '1';
			$query['payment_default'] = 0;
			Viveum_General_Models::viveum_update_db_default_payment( $query, $payment_group, $credentials );

			$query['field'] = 'id';
			$query['value'] = $recurring_id;
			$query['payment_default'] = 1;
			Viveum_General_Models::viveum_update_db_default_payment( $query, $payment_group, $credentials );
		}
		wp_safe_redirect( self::$clean_current_url );
	}

	/**
	 * Redirect error
	 *
	 * @param string $error_identifier error identifier.
	 * @param string $error_default error default.
	 */
	private static function viveum_redirect_error( $error_identifier, $error_default = 'ERROR_UNKNOWN' ) {
		if ( 'ERROR_UNKNOWN' === $error_identifier ) {
			$error_identifier = $error_default;
		}
		$error_translated = Viveum_General_Functions::viveum_translate_error_identifier( $error_identifier );
		WC()->session->set( 'viveum_myaccount_error', $error_translated );
		wp_safe_redirect( self::$clean_current_url );
		exit();
	}

	/**
	 * Load register payment account page
	 * Get chekout id & Payment widget from opp
	 *
	 * @param string $payment_id payment id.
	 */
	private static function viveum_register_payment( $payment_id ) {
		$register_parameters = self::viveum_get_register_parameters( $payment_id, true );
		$register_parameters['merchantTransactionId'] = get_current_user_id();

		$checkout_result = ViveumPaymentCore::get_checkout_result( $register_parameters );

		if ( ! $checkout_result['is_valid'] ) {
			self::viveum_redirect_error( $checkout_result['response'] );
		} elseif ( ! isset( $checkout_result['response']['id'] ) ) {
			self::viveum_redirect_error( 'ERROR_GENERAL_REDIRECT' );
		}

		$url_config['payment_widget'] = ViveumPaymentCore::get_payment_widget_url( $register_parameters['server_mode'], $checkout_result['response']['id'] );
		$url_config['return_url'] = self::$current_url . 'viveum_page=wc-response-register&viveum_section=' . $payment_id;
		$url_config['cancel_url'] = self::$clean_current_url;

		$form_title = __( 'Save Payment Information', 'wc-viveum' );
		$confirm_button = __( 'Register', 'wc-viveum' );

		$payment_gateway = Viveum_General_Functions::viveum_get_payment_gateway_by_id( $payment_id );
		$args = array(
			'payment_id' => $payment_id,
			'plugin_url' => self::$plugin_url,
			'url_config' => $url_config,
			'form_title' => $form_title,
			'confirm_button' => $confirm_button,
			'register_parameters' => $register_parameters,
			'payment_brand' => $payment_gateway->get_payment_brand(),
		);
		wp_enqueue_script( 'viveum_formpayment_script', $url_config['payment_widget'], array(), null );
		wp_enqueue_style( 'viveum_formpayment_style', self::$plugin_url . '/assets/css/formpayment.css', array(), null );
		Viveum_General_Functions::viveum_include_template( self::$plugin_path . '/templates/myaccount/template-register.php', $args );
	}

	/**
	 * Load change ( reregisster ) payment account page
	 * Get chekout id & payment widget from opp
	 *
	 * @param string $payment_id payment id.
	 */
	private static function viveum_change_payment( $payment_id ) {
		$id = Viveum_General_Functions::viveum_get_request_value( 'id' );

		$registration_id = Viveum_General_Models::viveum_get_db_registration_id( $id );
		$register_parameters = self::viveum_get_register_parameters( $payment_id, true );
		$register_parameters['merchantTransactionId'] = $registration_id;

		$checkout_result = ViveumPaymentCore::get_checkout_result( $register_parameters );

		if ( ! $checkout_result['is_valid'] ) {
			self::viveum_redirect_error( $checkout_result['response'] );
		} elseif ( ! isset( $checkout_result['response']['id'] ) ) {
			self::viveum_redirect_error( 'ERROR_GENERAL_REDIRECT' );
		}

		$url_config['payment_widget'] = ViveumPaymentCore::get_payment_widget_url( $register_parameters['server_mode'], $checkout_result['response']['id'] );
		$url_config['return_url'] = self::$current_url . 'viveum_page=wc-response-reregister&viveum_section=' . $payment_id . '&recurring_id=' . $id;
		$url_config['cancel_url'] = self::$clean_current_url;

		$form_title = __( 'Change Payment Information', 'wc-viveum' );
		$confirm_button = __( 'Change', 'wc-viveum' );

		$payment_gateway = Viveum_General_Functions::viveum_get_payment_gateway_by_id( $payment_id );
		$args = array(
			'payment_id' => $payment_id,
			'plugin_url' => self::$plugin_url,
			'url_config' => $url_config,
			'form_title' => $form_title,
			'confirm_button' => $confirm_button,
			'register_parameters' => $register_parameters,
			'payment_brand' => $payment_gateway->get_payment_brand(),
		);
		wp_enqueue_script( 'viveum_formpayment_script', $url_config['payment_widget'], array(), null );
		wp_enqueue_style( 'viveum_formpayment_style', self::$plugin_url . '/assets/css/formpayment.css', array(), null );
		Viveum_General_Functions::viveum_include_template( self::$plugin_path . '/templates/myaccount/template-register.php', $args );
	}

	/**
	 * Load delete ( deregisster ) payment account page
	 *
	 * @param string $payment_id payment id.
	 */
	private static function viveum_delete_payment( $payment_id ) {
		$id = Viveum_General_Functions::viveum_get_request_value( 'id' );
		$url_config['return_url'] = self::$current_url . 'viveum_page=wc-response-deregister&viveum_section=' . $payment_id . '&recurring_id=' . $id;
		$url_config['cancel_url'] = self::$clean_current_url;

		$args = array(
			'plugin_url' => self::$plugin_url,
			'url_config' => $url_config,
		);

		wp_enqueue_style( 'viveum_formpayment_style', self::$plugin_url . '/assets/css/formpayment.css', array(), null );
		Viveum_General_Functions::viveum_include_template( self::$plugin_path . '/templates/myaccount/template-deregister.php', $args );
	}

	/**
	 * Response from register/reregister payment account
	 * Get status & response from opp
	 *
	 * @param string $payment_id payment id.
	 * @param string $register_method register method.
	 */
	private static function viveum_response_register( $payment_id, $register_method ) {
		$payment_gateway = Viveum_General_Functions::viveum_get_payment_gateway_by_id( $payment_id );
		$checkout_id = Viveum_General_Functions::viveum_get_request_value( 'id' );
		$recurring_id = Viveum_General_Functions::viveum_get_request_value( 'recurring_id' );

		$is_testmode_available = false;
		$credentials = $payment_gateway->get_credentials( $is_testmode_available );
		$register_result = ViveumPaymentCore::get_payment_status( $checkout_id, $credentials );

		if ( ! $register_result['is_valid'] ) {
			self::viveum_redirect_error( $register_result['response'], 'ERROR_GENERAL_NORESPONSE' );
		} else {
			$register_status = ViveumPaymentCore::get_transaction_result( $register_result['response']['result']['code'] );

			Viveum_General_Functions::add_log( 'Register payment result ', $register_result );

			if ( 'ACK' === $register_status ) {
				self::viveum_do_success_register( $payment_id, $register_result['response'], $recurring_id );
				if ( 'register' === $register_method ) {

					Viveum_General_Functions::add_log( 'Register payment is success' );

					$success_message = __( 'Congratulations, your payment information were successfully saved.', 'wc-viveum' );
				} else {

					Viveum_General_Functions::add_log( 'Change payment is success' );

					$success_message = __( 'Congratulations, your payment information were successfully updated.', 'wc-viveum' );
				}

				WC()->session->set( 'viveum_myaccount_success', $success_message );
			} else {
				if ( 'register' === $register_method ) {

					Viveum_General_Functions::add_log( 'Register payment is failed' );

					$error_message = __( 'We are sorry. Your attempt to save your payment information was not successful.', 'wc-viveum' );
				} else {

					Viveum_General_Functions::add_log( 'Change payment is failed' );

					$error_message = __( 'We are sorry. Your attempt to update your payment information was not successful.', 'wc-viveum' );
				}

				WC()->session->set( 'viveum_myaccount_error', $error_message );
			}
			wp_safe_redirect( self::$clean_current_url );
		}// End if().
	}

	/**
	 * Success register
	 * Action if register is success
	 *
	 * @param string $payment_id payment id.
	 * @param array  $register_result payment id.
	 * @param string $recurring_id recurring_id.
	 */
	private static function viveum_do_success_register( $payment_id, $register_result, $recurring_id ) {
		$payment_gateway = Viveum_General_Functions::viveum_get_payment_gateway_by_id( $payment_id );
		$deregister_id = $register_result['merchantTransactionId'];
		$reference_id = $register_result['id'];

		$register_parameters = self::viveum_get_register_parameters( $payment_id );
		if ( $recurring_id ) {
			$register_parameters['transaction_id'] = $register_result['merchantTransactionId'];
		} else {
			$register_parameters['transaction_id'] = get_current_user_id();
		}

		if ( 'viveum_paypalsaved' === $payment_id ) {
			$registration_id = $register_result['id'];
			$register_parameters['payment_type'] = $payment_gateway->get_payment_type();
			$debit_result = self::viveum_debit_registered_paypal( $payment_id, $registration_id, $register_parameters );
			$reference_id = $debit_result['response']['id'];
		} else {
			$registration_id = $register_result['registrationId'];
		}

		self::viveum_refund_registered_payment( $registration_id, $reference_id, $register_parameters, $recurring_id );

		if ( $recurring_id ) {
			ViveumPaymentCore::delete_registered_account( $deregister_id, $register_parameters );
		}

		self::viveum_save_registered_payment( $payment_id, $register_result, $recurring_id );
	}

	/**
	 * Response from deregister payment
	 * Deregister & get response from opp
	 *
	 * @param string $payment_id payment id.
	 */
	private static function viveum_response_deregister( $payment_id ) {
		$payment_gateway = Viveum_General_Functions::viveum_get_payment_gateway_by_id( $payment_id );
		$recurring_id = Viveum_General_Functions::viveum_get_request_value( 'recurring_id', 0 );
		$registration_id = Viveum_General_Models::viveum_get_db_registration_id( $recurring_id );

		$register_parameters = $payment_gateway->get_credentials();
		$register_parameters['transaction_id'] = get_current_user_id();

		$deregister_result = ViveumPaymentCore::delete_registered_account( $registration_id, $register_parameters );

		Viveum_General_Functions::add_log( 'Deregister payment result ', $deregister_result );

		if ( ! $deregister_result['is_valid'] ) {
			Viveum_General_Functions::add_log( 'Deregister payment is failed : ' . $deregister_result['response'] );
			self::viveum_redirect_error( $deregister_result['response'], 'ERROR_MC_DELETE' );
		} else {
			$deregister_status = ViveumPaymentCore::get_transaction_result( $deregister_result['response']['result']['code'] );
			if ( 'ACK' === $deregister_status ) {

				Viveum_General_Functions::add_log( 'Deregister payment is success' );

				Viveum_General_Models::viveum_delete_db_registered_payment( $recurring_id );
				WC()->session->set( 'viveum_myaccount_success', __( 'Congratulations, your payment information were successfully deleted.', 'wc-viveum' ) );
			} else {

				Viveum_General_Functions::add_log( 'Deregister payment is failed' );

				WC()->session->set( 'viveum_myaccount_error', __( 'We are sorry. Your attempt to delete your payment information was not successful.', 'wc-viveum' ) );
			}
		}

		wp_safe_redirect( self::$clean_current_url );
	}

	/**
	 * Debit registered paypal
	 * Paypal initial debit after register payment
	 *
	 * @param string $payment_id payment id.
	 * @param int    $registration_id registration id.
	 * @param array  $register_parameters registration parameters.
	 * @return : $reference_id ( if success )
	 *         or
	 *         redirect error ( if failed )
	 */
	private static function viveum_debit_registered_paypal( $payment_id, $registration_id, $register_parameters ) {
		$register_parameters['payment_recurring'] = 'INITIAL';
		$paypal_result = ViveumPaymentCore::use_registered_account( $registration_id, $register_parameters );

		if ( ! $paypal_result['is_valid'] ) {
			self::viveum_delete_registered_account( $payment_id, $registration_id );
			self::viveum_redirect_error( $paypal_result['response'], 'ERROR_MC_ADD' );
		} else {
			$paypal_status = ViveumPaymentCore::get_transaction_result( $paypal_result['response']['result']['code'] );

			if ( 'ACK' === $paypal_status ) {
				return $paypal_result;
			} else {
				self::viveum_delete_registered_account( $payment_id, $registration_id );
				self::viveum_redirect_error( 'ERROR_MC_ADD' );
			}
		}
	}

	/**
	 * Delete registered account
	 *
	 * @param string $payment_id payment id.
	 * @param int    $registration_id registration id.
	 */
	private static function viveum_delete_registered_account( $payment_id, $registration_id ) {
		$payment_gateway = Viveum_General_Functions::viveum_get_payment_gateway_by_id( $payment_id );
		Viveum_General_Models::viveum_delete_db_registered_payment_by_regid( $registration_id );
		$register_parameters = $payment_gateway->get_credentials();
		ViveumPaymentCore::delete_registered_account( $registration_id, $register_parameters );
	}

	/**
	 * Refund after success register payment account
	 * Refund & Capture/Reversal Amount from success register payment account
	 *
	 * @param int   $registration_id registration id.
	 * @param int   $reference_id reference id.
	 * @param array $register_parameters register parameters.
	 * @param array $recurring_id recurring id.
	 */
	private static function viveum_refund_registered_payment( $registration_id, $reference_id, $register_parameters, $recurring_id ) {
		$payment_type = $register_parameters['payment_type'];
		$register_parameters['payment_type'] = ( 'DB' === $payment_type )? 'RF' : 'CP';
		$backoffice_result = ViveumPaymentCore::back_office_operation( $reference_id, $register_parameters );

		if ( 'PA' === $payment_type ) {
			if ( $recurring_id ) {
				$error_default = 'ERROR_MC_UPDATE';
			} else {
				$error_default = 'ERROR_MC_ADD';
			}
			if ( ! $backoffice_result['is_valid'] ) {
				ViveumPaymentCore::delete_registered_account( $registration_id, $register_parameters );
				self::viveum_redirect_error( $backoffice_result['response'], $error_default );
			} else {
				$transaction_result = ViveumPaymentCore::get_transaction_result( $backoffice_result['response']['result']['code'] );
				if ( 'ACK' === $transaction_result ) {
					$reference_id = $backoffice_result['response']['id'];
					$register_parameters['payment_type'] = 'RF';
					ViveumPaymentCore::back_office_operation( $reference_id, $register_parameters );
				} else {
					ViveumPaymentCore::delete_registered_account( $registration_id, $register_parameters );
					self::viveum_redirect_error( $error_default );
				}
			}
		}
	}

	/**
	 * Get parameter for register payment account
	 *
	 * @param string $payment_id payment id.
	 * @param bool   $registration recuring or not.
	 * @return $register_parameters
	 */
	private static function viveum_get_register_parameters( $payment_id, $registration = false ) {
		$payment_gateway = Viveum_General_Functions::viveum_get_payment_gateway_by_id( $payment_id );
		$payment_settings = get_option( 'woocommerce_' . $payment_id . '_settings' );
		$register_parameters = $payment_gateway->get_credentials();

		$register_parameters['amount']         = $payment_settings['amount_registration'];
		$register_parameters['currency']       = get_woocommerce_currency();
		$register_parameters['payment_type']   = $payment_gateway->get_payment_type();

		if ( $registration ) {
			$register_parameters['payment_recurring'] = 'INITIAL';
			$register_parameters['payment_registration'] = 'true';

			if ( 'viveum_paypalsaved' === $payment_id ) {
				unset( $register_parameters['payment_type'] );
			}
		}

		if ( 'viveum_ccsaved' === $payment_id ) {
			$register_parameters['3D']['amount']    = $register_parameters['amount'] ;
			$register_parameters['3D']['currency']  = $register_parameters['currency'];
		}

		$register_parameters['customer']['first_name']   = esc_attr( get_the_author_meta( 'billing_first_name', get_current_user_id() ) );
		$register_parameters['customer']['last_name']     = esc_attr( get_the_author_meta( 'billing_last_name', get_current_user_id() ) );
		$register_parameters['customer']['phone']         = esc_attr( get_the_author_meta( 'billing_phone', get_current_user_id() ) );
		$register_parameters['customer']['email']         = esc_attr( get_the_author_meta( 'billing_email', get_current_user_id() ) );
		$register_parameters['billing']['street']         = esc_attr( get_the_author_meta( 'billing_address_1', get_current_user_id() ) );
		$register_parameters['billing']['city']         = esc_attr( get_the_author_meta( 'billing_city', get_current_user_id() ) );
		$register_parameters['billing']['zip']           = esc_attr( get_the_author_meta( 'billing_postcode', get_current_user_id() ) );
		$register_parameters['billing']['country_code'] = esc_attr( get_the_author_meta( 'billing_country', get_current_user_id() ) );

		$register_parameters['customer_ip'] = Viveum_General_Functions::get_customer_ip();
		
		$register_parameters['customParameters'] = array(
			'viveum_payment_id' => $payment_id,
		);

		return $register_parameters;
	}

	/**
	 * Save registered payment account to db
	 *
	 * @param string $payment_id payment id.
	 * @param array  $register_result register result.
	 * @param int    $recurring_id recurring id.
	 */
	private static function viveum_save_registered_payment( $payment_id, $register_result, $recurring_id ) {
		global $wpdb;

		$payment_gateway = Viveum_General_Functions::viveum_get_payment_gateway_by_id( $payment_id );
		$payment_group = $payment_gateway->get_payment_group();
		$credentials = $payment_gateway->get_credentials();
		$registered_payment = Viveum_General_Functions::viveum_get_account_by_result( $payment_id, $register_result );

		$registered_payment['cust_id']         = get_current_user_id();
		$registered_payment['payment_brand']   = $register_result['paymentBrand'];
		$registered_payment['payment_group']   = $payment_group;
		$registered_payment['payment_default'] = Viveum_General_Models::viveum_get_db_payment_default( $payment_group, $credentials );
		$registered_payment['server_mode']     = $credentials['server_mode'];
		$registered_payment['channel_id']      = $credentials['channel_id'];

		if ( 'viveum_paypalsaved' === $payment_id ) {
			$registered_payment['registration_id'] = $register_result['id'];
		} else {
			$registered_payment['registration_id'] = $register_result['registrationId'];
		}

		if ( $recurring_id ) {
			Viveum_General_Models::viveum_update_db_registered_payment( $recurring_id, $registered_payment );
		} else {
			Viveum_General_Models::viveum_save_db_registered_payment( $registered_payment );
		}
	}

}

add_shortcode( 'woocommerce_my_payment_information', 'Viveum_Payment_Information::init' );
